<?php

/**
 * cardsave
 *
 * @author     Conor Kerr <cardsave@dev.ceon.net>
 * @copyright  Copyright 2008 Ceon
 * @copyright  Portions Zen to OS-Commerce Conversion 22/03/2010 - David McCann, CardSave Online 
 * @copyright  Portions Copyright 2003-2006 Zen Cart Development Team
 * @copyright  Portions Copyright 2003 osCommerce
 * @link       http://dev.ceon.net/web/zen-cart/cardsave
 * @license    http://www.gnu.org/copyleft/gpl.html   GNU Public License V2.0
 * @version    $Id: cardsave.php 745 2008-09-12 10:10:31Z conor $
 */

/**
 * Version definition, don't touch!
 */
define('MODULE_PAYMENT_CARDSAVE_VERSION_NO', '1.4.3');


// {{{ cardsave

/**
 * Payment Module conforming to Zen Cart format. Retains all Card Details entered throughout
 * the checkout process, making use of PEAR Crypt Blowfish, if possible, to encrypt the details and
 * therefore hopefully comply with any applicable Data Protection Laws.
 *
 * @author     Conor Kerr <cardsave@dev.ceon.net>
 * @copyright  Copyright 2008 Ceon
 * @copyright  Portions Copyright 2003-2006 Zen Cart Development Team
 * @copyright  Portions Copyright 2003 osCommerce
 * @link       http://dev.ceon.net/web/zen-cart/cardsave
 * @license    http://www.gnu.org/copyleft/gpl.html   GNU Public License V2.0
 */
class cardsave
{
	// {{{ properties
	
	/**
	 * The internal 'code' name used to designate "this" payment module.
	 *
	 * @var     string
	 * @access  public
	 */
	var $code;
	
	/**
	 * The version of this module.
	 *
	 * @var     string
	 * @access  public
	 */
	var $version;
	
	/**
	 * The name displayed for this payment method.
	 *
	 * @var     string
	 * @access  public
	 */
	var $title;
	
	/**
	 * The description displayed for this payment method.
	 *
	 * @var     string
	 * @access  public
	 */
	var $description;
	
	/**
	 * Module status - set based on various config and zone criteria.
	 *
	 * @var     boolean
	 * @access  public
	 */
	var $enabled;
	
	/**
	 * The zone to which this module is restricted for use.
	 *
	 * @var     integer
	 * @access  public
	 */
	var $zone;
	
	/**
	 * The sort order of display for this module within the checkout's payment method listing.
	 *
	 * @var     integer
	 * @access  public
	 */
	var $sort_order;
	
	/**
	 * The order status setting for orders which have been passed to CardSave.
	 *
	 * @var     integer
	 * @access  public
	 * @default 0
	 */
	var $order_status = 0;
	
	/**
	 * The values for the data returned by cardsave from a transaction query.
	 *
	 * @var     array
	 * @access  protected
	 */
	var $_cardsave_return_values;
	
	// }}}

	
	// {{{ Class Constructor
	
	/**
	 * Create a new instance of the cardsave class
	 * 
	 * @access  public
	 * @param   none
	 */
	function cardsave()
	{
		global $order, $db;
		
		$this->code = 'cardsave';
		$this->version = MODULE_PAYMENT_CARDSAVE_VERSION_NO;
		
		// Perform error checking of module's configuration ////////////////////////////////////////
		
		// Variable holds status of configuration checks so that module can be disabled if it cannot
		// perform its function
		$critical_config_problem = false;
		
		$cardsave_config_messages = '';
		
		// Output warning if database table doesn't exist
		$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CARDSAVE . '";';
		
    //$table_exists_result = $db->Execute($table_exists_query);
    $my_var_query = tep_db_query($table_exists_query);
		$table_exists_result = tep_db_fetch_array($my_var_query); 
    
		if ($table_exists_result->EOF) {
			// Database doesn't exist
			$critical_config_problem = true;
			
			$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />The CardSave Database Table Does Not Exist!</strong><br /><br /><strong><span style="color: red">Please create the database table, according to the installation instructions!</span></strong><br /><br /><br />';
		}
		
		if (!defined('MODULE_PAYMENT_CARDSAVE_STORE_DETAILS_IN_SESSION') ||
				MODULE_PAYMENT_CARDSAVE_STORE_DETAILS_IN_SESSION == 'Yes') {
			// Output warning if Blowfish encryption class not found
			if (!$this->_file_exists_in_include_path('Crypt/Blowfish.php')) {
				$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />You do NOT have PEAR:Crypt_Blowfish installed on your server!</strong><br /><br /><span style="color: red">Encryption will not be used for the Credit/Debit Card Details if they are being stored in the Session.</span><br /><br /><strong>Please consult the documentation (especially the FAQs) to see how to resolve this.</strong><br /><br /><br />';
			}
			
			// Output warning if PEAR files not found
			if (!$this->_file_exists_in_include_path('PEAR.php')) {
				$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />Your PHP installation does NOT have access to PEAR  on your server!</strong><br /><br /><span style="color: red">Blowfish encryption cannot be used for any Credit/Debit Card Details being stored in the Session.</span><br /><br /><strong>Please consult the documentation (especially the FAQs) to see how to resolve this.</strong><br /><br /><br />';
			}
		}
		
		// Output warning if surcharge/discount functionality enabled but module not installed
		if ((defined('MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS') &&
				MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS == 'Yes')
				&& (!defined('MODULE_ORDER_TOTAL_PAYMENT_SURCHARGES_DISCOUNTS_ENABLED')
				|| MODULE_ORDER_TOTAL_PAYMENT_SURCHARGES_DISCOUNTS_ENABLED != 'Yes')) {
			$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />Payment Surcharges/Discounts module not installed!</strong><br /><br /><strong>You have enabled the Surcharge/Discount functionality but have not installed/enabled the Payment Surcharges/Discounts Order Total module!</strong><br /><br /><br />';
		}
		
		// Output warning if cURL functions not found
		if (!extension_loaded('curl')) {
			$critical_config_problem = true;
			
			$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />You do NOT have the cURL extension loaded in your PHP installation!</strong><br /><br /><span style="color: red">The module cannot be used without this extension.<br /><br /><br />';
		}
		
		// Extra checks can be performed only when module is installed
		if (defined(MODULE_PAYMENT_CARDSAVE_ACCEPTED_CURRENCIES)) {
			// Output warning message if shop's default currency has no matching merchant
			// account
			if (strpos(MODULE_PAYMENT_CARDSAVE_ACCEPTED_CURRENCIES, 
					DEFAULT_CURRENCY) === false) {
				$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />Your shop\'s Default Currency does not have a matching Merchant Account!<br /><br /> If the customer is browsing the shop in a currency for which you have no matching merchant account, all prices will be converted to the Default Merchant Account\'s Currency upon checkout. <br /><br />This would mean that what the customer will be charged will be likely to differ from what their order says they\'ll be charged - this is not recommended!<br /><br />You should use a Merchant Account which can accept your shop\'s Default Currency!</strong><br /><br /><br />';
			}
			
			// Output warning if the default merchant account's currency isn't in the list of
			// accepted currencies
			if (strpos(MODULE_PAYMENT_CARDSAVE_ACCEPTED_CURRENCIES, 
					MODULE_PAYMENT_CARDSAVE_DEFAULT_CURRENCY) === false) {
				$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />Your Default Merchant Account\'s Currency is not in the list of Accepted Currencies!</strong><br /><br /><br />';
			}
			
			// Output warning if the default merchant account's currency isn't used by the
			// shop... any potential conversions could not take place if it isn't!
			if (!class_exists('currencies')) {
				require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'currencies.php');
			}
			$check_currencies = new currencies();
			
			// Make sure currencies class was found
			if (!is_a($check_currencies, 'currencies')) {
				$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />Could not load the currencies class to determine if your Default Merchant Account Currency is used by your shop!<br /><br />Check the path to the currencies class in includes/languages/english/modules/payment/cardsave_form.php!</strong><br /><br /><br />';
			} else {
				$currency_not_used = false;
				if (method_exists($check_currencies, 'is_set')) {
					if (!$check_currencies->is_set(MODULE_PAYMENT_CARDSAVE_DEFAULT_CURRENCY)) {
						$currency_not_used = true;
					}
				} else {
					// Check the currency manually as Zen Cart's admin currencies class lacks
					// some functionality
					if (!isset($check_currencies->currencies[MODULE_PAYMENT_CARDSAVE_DEFAULT_CURRENCY]) || 
							!tep_not_null($check_currencies->currencies[MODULE_PAYMENT_CARDSAVE_DEFAULT_CURRENCY])) {
						$currency_not_used = true;
					}
				}
				if ($currency_not_used) {
					$critical_config_problem = true;
					
					$cardsave_config_messages .= '<strong><span style="color: red">Warning:</span><br />Your Default Merchant Account\'s Currency isn\'t used by your shop - currency conversions cannot take place!<br /><br />You must add a matching currency in the shop admin!</strong><br /><br /><br />';
				}
			}
		}
		
		// Check the database version of the module against the version of the files
		$cardsave_config_messages .= '<fieldset style="background: #F7F6F0; margin-bottom: 1.5em"><legend style="font-size: 1.2em; font-weight: bold">Module Version Information</legend>';
		if (defined('MODULE_PAYMENT_CARDSAVE_MADE_BY_CEON')) {
			$cardsave_database_version = MODULE_PAYMENT_CARDSAVE_MADE_BY_CEON;
			
			if ($cardsave_database_version == '1.4.0' ||
					$cardsave_database_version == '1.4.1'||
					$cardsave_database_version == '1.4.2') {
				$cardsave_database_version = $this->version;
			}
		} else {
			// Module not installed!
			$cardsave_database_version = null;
		}
		if (!is_null($cardsave_database_version) &&
				$cardsave_database_version != $this->version) {
			// Database version doesn't match expected version
			$critical_config_problem = true;
			
			$cardsave_config_messages .= '<p style="color: red"><strong>Module is out of date!</strong></p><p style="color: red">Please &ldquo;Remove&rdquo; and &ldquo;Install&rdquo; the module as per the upgrade instructions that come with this module!</p>';
		}
		
		$cardsave_config_messages .= '<p>File Version: ' . $this->version;
		$cardsave_config_messages .= '<br />Database Version: ' . (is_null($cardsave_database_version) ? 'Module Not Installed' : $cardsave_database_version) . '</p>';
		
		// Set the title and description based on the mode the module is in: Admin or Catalog
		if ((defined('IS_ADMIN_FLAG') && IS_ADMIN_FLAG === true) || (!isset($_GET['main_page']) ||
				$_GET['main_page'] == '')) {
			// In Admin mode
			$this->title = sprintf(MODULE_PAYMENT_CARDSAVE_TEXT_ADMIN_TITLE, $this->version);
			$this->description = $cardsave_config_messages . 
				MODULE_PAYMENT_CARDSAVE_TEXT_DESCRIPTION_BASE;
		} else {
			// In Catalog mode
			$this->title = MODULE_PAYMENT_CARDSAVE_TEXT_CATALOG_TITLE;
			$this->description = '';
		}
		
		// Disable the module if configured as such or a critical configuration error was found
		$this->enabled = ((defined('MODULE_PAYMENT_CARDSAVE_STATUS') &&
			MODULE_PAYMENT_CARDSAVE_STATUS == 'Yes' &&
			$critical_config_problem == false) ? true : false);
		
		if (defined('MODULE_PAYMENT_CARDSAVE_SORT_ORDER')) {
			$this->sort_order = MODULE_PAYMENT_CARDSAVE_SORT_ORDER;
		}
		
		if (defined('MODULE_PAYMENT_CARDSAVE_ORDER_STATUS_ID') &&
				(int) MODULE_PAYMENT_CARDSAVE_ORDER_STATUS_ID > 0) {
			$this->order_status = MODULE_PAYMENT_CARDSAVE_ORDER_STATUS_ID;
		}
		
		if (defined('MODULE_PAYMENT_CARDSAVE_ZONE')) {
			$this->zone = (int) MODULE_PAYMENT_CARDSAVE_ZONE;
		}
		
		if (is_object($order)) {
			$this->update_status();
		}
	}
	
	// }}}
	
	
	// {{{ update_status()

	/**
	 * Determines whether or not this payment method should be used for the current zone.
	 *
	 * @access  public
	 * @param   none
	 * @return  none
	 */
	function update_status()
	{
		global $order, $db;
		
		if (($this->enabled == true) && ($this->zone > 0)) {
			$check_flag = false;
			$sql = "
				SELECT
					zone_id
				FROM
					" . TABLE_ZONES_TO_GEO_ZONES . "
				WHERE
					geo_zone_id = '" . $this->zone . "'
				AND
					zone_country_id = '" . $order->billing['country']['id'] . "'
				ORDER BY
					zone_id
				";
			
			//$check = $db->Execute($sql);
      $my_var_query = tep_db_query($sql);
		  $check = tep_db_fetch_array($my_var_query);
			while (!$check->EOF) {
				if ($check->fields['zone_id'] < 1) {
					$check_flag = true;
					break;
				} elseif ($check->fields['zone_id'] == $order->billing['zone_id']) {
					$check_flag = true;
					break;
				}
				$check->MoveNext();
			}
			
			if ($check_flag == false) {
				$this->enabled = false;
			}
		}
	}
	
	// }}}
	
	
	// {{{ javascript_validation()

	/**
	 * Validates the Card Details via Javascript (Number, Owner, Type and CV2 Length)
	 *
	 * @access  public
	 * @param   none
	 * @return  string   The Javascript needed to check the submitted Card details.
	 */
	function javascript_validation()
	{
		$js = '  if (payment_value == "' . $this->code . '") {' . "\n" .
			'    var cardsave_error_class = "CardSaveFormGadgetError";' . "\n" .
			'    var cardsave_card_owner_gadget = document.checkout_payment.cardsave_card_owner;' . "\n" .
			'    var cardsave_card_number_gadget = document.checkout_payment.cardsave_card_number;' . "\n" .
			'    var cardsave_card_type_gadget = document.checkout_payment.cardsave_card_type;' . "\n" .
			'    var cardsave_card_type_gadget_value = cardsave_card_type_gadget.options[cardsave_card_type_gadget.selectedIndex].value;' . "\n" .
			'    var cardsave_card_expires_month_gadget = document.checkout_payment.cardsave_card_expires_month;' . "\n" .
			'    var cardsave_card_expires_month_gadget_value = cardsave_card_expires_month_gadget.options[cardsave_card_expires_month_gadget.selectedIndex].value;' . "\n" .
			'    var cardsave_card_expires_year_gadget = document.checkout_payment.cardsave_card_expires_year;' . "\n" .
			'    var cardsave_card_expires_year_gadget_value = cardsave_card_expires_year_gadget.options[cardsave_card_expires_year_gadget.selectedIndex].value;' . "\n";
		
		if (MODULE_PAYMENT_CARDSAVE_USE_CV2 == 'Yes')  {
			$js .= '    var cardsave_card_cv2_gadget = document.checkout_payment.cardsave_card_cv2;' . "\n";
		}
		
		if ($this->_useStartDate()) {
			$js .= '    var cardsave_card_start_month_gadget = document.checkout_payment.cardsave_card_start_month;' . "\n" .
			'    var cardsave_card_start_month_gadget_value = cardsave_card_start_month_gadget.options[cardsave_card_start_month_gadget.selectedIndex].value;' . "\n" .
			'    var cardsave_card_start_year_gadget = document.checkout_payment.cardsave_card_start_year;' . "\n" .
			'    var cardsave_card_start_year_gadget_value = cardsave_card_start_year_gadget.options[cardsave_card_start_year_gadget.selectedIndex].value;' . "\n";
		}
		
		if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes') {
			$js .= '    var cardsave_card_test_case_gadget = document.checkout_payment.cardsave_use_test_case;' . "\n";
			$js .= ' if (cardsave_use_test_case.selectedIndex == 0) {' . "\n";
		}
		
		$js .= '    if (cardsave_card_owner_gadget.value == "" || cardsave_card_owner_gadget.value.length < ' . CC_OWNER_MIN_LENGTH . ') {' . "\n" .
			'      error_message = error_message + "' . MODULE_PAYMENT_CARDSAVE_TEXT_JS_CARD_OWNER . '";' . "\n" .
			'      error = 1;' . "\n" .
			'      // Update the form gadget\'s class to give visual feedback to user' . "\n" .
			'      if (cardsave_card_owner_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
			'        cardsave_card_owner_gadget.className =  cardsave_card_owner_gadget.className + " " + cardsave_error_class;' . "\n" .
			'      }' . "\n" .
			'    } else {' . "\n" .
			'      // Reset error status if necessary' . "\n" .
			'      cardsave_card_owner_gadget.className = cardsave_card_owner_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'    }' . "\n" .
			'    if (cardsave_card_type_gadget_value == "xxx") {' . "\n" .
			'      error_message = error_message + "' . MODULE_PAYMENT_CARDSAVE_TEXT_JS_CARD_TYPE . '";' . "\n" .
			'      error = 1;' . "\n" .
			'      // Update the form gadget\'s class to give visual feedback to user' . "\n" .
			'      if (cardsave_card_type_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
			'        cardsave_card_type_gadget.className =  cardsave_card_type_gadget.className + " " + cardsave_error_class;' . "\n" .
			'      }' . "\n" .
			'    } else {' . "\n" .
			'      // Reset error status if necessary' . "\n" .
			'      cardsave_card_type_gadget.className = cardsave_card_type_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'    }' . "\n" .
			'    if (cardsave_card_number_gadget.value == "" || cardsave_card_number_gadget.value.length < ' . CC_NUMBER_MIN_LENGTH . ') {' . "\n" .
			'      error_message = error_message + "' . MODULE_PAYMENT_CARDSAVE_TEXT_JS_CARD_NUMBER . '";' . "\n" .
			'      error = 1;' . "\n" .
			'      // Update the form gadget\'s class to give visual feedback to user' . "\n" .
			'      if (cardsave_card_number_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
			'        cardsave_card_number_gadget.className =  cardsave_card_number_gadget.className + " " + cardsave_error_class;' . "\n" .
			'      }' . "\n" .
			'    } else {' . "\n" .
			'      // Reset error status if necessary' . "\n" .
			'      cardsave_card_number_gadget.className = cardsave_card_number_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'    }' . "\n" .
			'    if (cardsave_card_expires_month_gadget_value == "" || cardsave_card_expires_year_gadget_value == "") {' . "\n" .
			'      error_message = error_message + "' . MODULE_PAYMENT_CARDSAVE_TEXT_JS_CARD_EXPIRY . '";' . "\n" .
			'      error = 1;' . "\n" .
			'    }' . "\n" .
			'    if (cardsave_card_expires_month_gadget_value == "") {' . "\n" .
			'      // Update the form gadget\'s class to give visual feedback to user' . "\n" .
			'      if (cardsave_card_expires_month_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
			'        cardsave_card_expires_month_gadget.className =  cardsave_card_expires_month_gadget.className + " " + cardsave_error_class;' . "\n" .
			'      }' . "\n" .
			'    } else {' . "\n" .
			'      // Reset error status if necessary' . "\n" .
			'      cardsave_card_expires_month_gadget.className = cardsave_card_expires_month_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'    }' . "\n" .
			'    if (cardsave_card_expires_year_gadget_value == "") {' . "\n" .
			'      // Update the form gadget\'s class to give visual feedback to user' . "\n" .
			'      if (cardsave_card_expires_year_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
			'        cardsave_card_expires_year_gadget.className =  cardsave_card_expires_year_gadget.className + " " + cardsave_error_class;' . "\n" .
			'      }' . "\n" .
			'    } else {' . "\n" .
			'      // Reset error status if necessary' . "\n" .
			'      cardsave_card_expires_year_gadget.className = cardsave_card_expires_year_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'    }' . "\n";
		
		if (MODULE_PAYMENT_CARDSAVE_USE_CV2 == 'Yes')  {
			$js .= '    if (cardsave_card_cv2_gadget.value == "" || cardsave_card_cv2_gadget.value.length < "3" || cardsave_card_cv2_gadget.value.length > "4") {' . "\n".
				'      error_message = error_message + "' . MODULE_PAYMENT_CARDSAVE_TEXT_JS_CARD_CV2 . '";' . "\n" .
				'      error = 1;' . "\n" .
				'      // Update the form gadget\'s class to give visual feedback to user' . "\n" .
				'      if (cardsave_card_cv2_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
				'        cardsave_card_cv2_gadget.className =  cardsave_card_cv2_gadget.className + " " + cardsave_error_class;' . "\n" .
				'      }' . "\n" .
				'    } else {' . "\n" .
				'      // Reset error status if necessary' . "\n" .
				'      cardsave_card_cv2_gadget.className = cardsave_card_cv2_gadget.className.replace(cardsave_error_class, "");' . "\n" .
				'    }' . "\n";
		}
		
		if ($this->_useStartDate()) {
			$js .=
			'    if ((cardsave_card_start_month_gadget_value == "" && cardsave_card_start_year_gadget_value != "")' . "\n" .
			'       || (cardsave_card_start_month_gadget_value != "" && cardsave_card_start_year_gadget_value == "")) {' . "\n" .
			'        error_message = error_message + "' . MODULE_PAYMENT_CARDSAVE_TEXT_JS_CARD_START . '";' . "\n" .
			'        error = 1;' . "\n" .
			'        if (cardsave_card_start_month_gadget_value == "") {' . "\n" .
			'          // Update the form gadget\'s class to give visual feedback to user' . "\n" .
			'          if (cardsave_card_start_month_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
			'            cardsave_card_start_month_gadget.className =  cardsave_card_start_month_gadget.className + " " + cardsave_error_class;' . "\n" .
			'          }' . "\n" .
			'        } else {' . "\n" .
			'          // Reset error status if necessary' . "\n" .
			'          cardsave_card_start_month_gadget.className = cardsave_card_start_month_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'        }' . "\n" .
			'        if (cardsave_card_start_year_gadget_value == "") {' . "\n" .
			'          // Update the form gadget\'s class to give visual feedback to user' . "\n" .
			'          if (cardsave_card_start_year_gadget.className.indexOf(cardsave_error_class) == -1) {' . "\n" .
			'            cardsave_card_start_year_gadget.className =  cardsave_card_start_year_gadget.className + " " + cardsave_error_class;' . "\n" .
			'          }' . "\n" .
			'        } else {' . "\n" .
			'          // Reset error status if necessary' . "\n" .
			'          cardsave_card_start_year_gadget.className = cardsave_card_start_year_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'        }' . "\n" .
			'    } else {' . "\n" .
			'        // Make sure that, if user hasn\'t used either start date field, they aren\'t marked as having an error' . "\n" .
			'        cardsave_card_start_month_gadget.className = cardsave_card_start_month_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'        cardsave_card_start_year_gadget.className = cardsave_card_start_year_gadget.className.replace(cardsave_error_class, "");' . "\n" .
			'    }' . "\n";
		}
		
		if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes') {
			$js .= ' }' . "\n";
		}
		
		$js .= '  }' . "\n";
		
		return $js;
	}
	
	// }}}
	
	
	// {{{ selection()

	/**
	 * Builds the Card Details Submission Fields for display on the Checkout Payment Page
	 *
	 * @access  public
	 * @param   none
	 * @return  array   The data needed to build the Card Details Submission Form.
	 *
	 *                  Array Format:
	 *
	 *                  id      string  The name of this payment class
	 *
	 *                  module  string  The title for this payment method
	 *
	 *                  fields  array   A list of the titles and form gadgets to be used to build
	 *                                  the form
	 *
	 *                                  Array Format:
	 *
	 *                 title    string  The title for this form field
	 *
	 *                 field    string  The HTML source for the form gadget for this form field
	 */
	function selection()
	{
		global $order;
		
		// Build the options for the Expiry and Start Date Select Gadgets //////////////////////////
		$expires_month[] = array('id' => '', 'text' => MODULE_PAYMENT_CARDSAVE_SELECT_MONTH);
		for ($i = 1; $i < 13; $i++) {
			$expires_month[] = array(
				'id' => sprintf('%02d', $i),
				'text' => strftime(MODULE_PAYMENT_CARDSAVE_SELECT_MONTH_FORMAT,
					mktime(0, 0, 0, $i, 1, 2000))
				);
		}
		
		// The Expiry Year options include the next ten years and this year
		$today = getdate();
		$expires_year[] = array('id' => '', 'text' => MODULE_PAYMENT_CARDSAVE_SELECT_YEAR);
		for ($i = $today['year']; $i < $today['year'] + 10; $i++) {
			$expires_year[] = array(
				'id' => strftime('%y', mktime(0, 0, 0, 1, 1, $i)),
				'text' => strftime(MODULE_PAYMENT_CARDSAVE_SELECT_YEAR_FORMAT,
					mktime(0, 0, 0, 1, 1, $i))
				);
		}
		
		$start_month[] = array('id' => '', 'text' => MODULE_PAYMENT_CARDSAVE_SELECT_MONTH);
		for ($i = 1; $i < 13; $i++) {
			$start_month[] = array(
				'id' => sprintf('%02d', $i),
				'text' => strftime(MODULE_PAYMENT_CARDSAVE_SELECT_MONTH_FORMAT,
					mktime(0,0,0,$i,1,2000))
				);
		}
		
		// The Start Year options include the past four years and this year
		$start_year[] = array('id' => '', 'text' => MODULE_PAYMENT_CARDSAVE_SELECT_YEAR);
		for ($i = $today['year'] - 4; $i <= $today['year']; $i++) {
			$start_year[] = array(
				'id' => strftime('%y', mktime(0, 0, 0, 1, 1, $i)),
				'text' => strftime(MODULE_PAYMENT_CARDSAVE_SELECT_YEAR_FORMAT,
					mktime(0, 0, 0, 1, 1, $i))
				);
		}
		
		// Build the options for the Card Type /////////////////////////////////////////////////////
		// Card Type selection is necessary for offering information about any Card
		// Surcharges/Discounts 
		$card_type[] = array('id' => 'xxx', 'text' => MODULE_PAYMENT_CARDSAVE_TEXT_SELECT_CARD_TYPE);
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA == 'Yes') {
			$card_type[] = array('id' => 'VISA', 'text' => $this->_getCardTypeNameForCode('VISA'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_MASTERCARD == 'Yes') {
			$card_type[] = array('id' => 'MASTERCARD', 'text' => $this->_getCardTypeNameForCode('MASTERCARD'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_DEBIT == 'Yes') {
			$card_type[] = array('id' => 'VISA_DEBIT', 'text' => $this->_getCardTypeNameForCode('VISA_DEBIT'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_SOLO == 'Yes') {
			$card_type[] = array('id' => 'SOLO', 'text' => $this->_getCardTypeNameForCode('SOLO'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_MAESTRO == 'Yes') {
			$card_type[] = array('id' => 'MAESTRO', 'text' => $this->_getCardTypeNameForCode('MAESTRO'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_ELECTRON == 'Yes') {
			$card_type[] = array('id' => 'VISA_ELECTRON', 'text' => $this->_getCardTypeNameForCode('VISA_ELECTRON'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_PURCHASING == 'Yes') {
			$card_type[] = array('id' => 'VISA_PURCHASING', 'text' => $this->_getCardTypeNameForCode('VISA_PURCHASING'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_AMERICAN_EXPRESS == 'Yes') {
			$card_type[] = array('id' => 'AMERICAN_EXPRESS', 'text' => $this->_getCardTypeNameForCode('AMERICAN_EXPRESS'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_DINERS_CLUB == 'Yes') {
			$card_type[] = array('id' => 'DINERS_CLUB', 'text' => $this->_getCardTypeNameForCode('DINERS_CLUB'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_JCB == 'Yes') {
			$card_type[] = array('id' => 'JCB', 'text' => $this->_getCardTypeNameForCode('JCB'));
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_LASER == 'Yes') {
			$card_type[] = array('id' => 'LASER', 'text' => $this->_getCardTypeNameForCode('LASER'));
		}
		
		// Initialise the default data to be used in the input form ////////////////////////////////
		$cardsave_card_owner = $order->billing['firstname'] . ' ' . $order->billing['lastname'];
		$cardsave_card_type = 'xxx';
		$cardsave_card_number = '';
		$cardsave_card_cv2 = '';
		$cardsave_card_expires_month = '';
		$cardsave_card_expires_year = '';
		$cardsave_card_issue = '';
		$cardsave_card_start_month = '';
		$cardsave_card_start_year = '';
		$cardsave_use_test_billing_address = true;
		
		if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes') {
			$cardsave_use_test_case = '-1';
		}
		
		// Check if the user has already entered their data. If so, use it to populate the form
		if (MODULE_PAYMENT_CARDSAVE_STORE_DETAILS_IN_SESSION == 'Yes' &&
				isset($_SESSION['cardsave_data_entered'])) {
			// Make sure that the user has been directly involved with the checkout process in the
			// previous step, otherwise this data should be considered expired
			$referring_uri = getenv("HTTP_REFERER");
			
			if ($referring_uri !== false
					&& strpos($referring_uri, 'main_page=checkout') === false
					&& strpos($referring_uri, 'main_page=shopping_cart') === false
					&& strpos($referring_uri, 'main_page=' . FILENAME_CHECKOUT_CARDSAVE_3D_SECURE) === false
					&& strpos($referring_uri, FILENAME_CARDSAVE_3D_SECURE_IFRAME) === false) {
				// Have not arrived here from another part of the checkout process or by a
				// redirect the result of a callback from a 3D-Secure check: data should be
				// considered invalid! Remove it from the session.
				unset($_SESSION['cardsave_data_entered']);
			} else {
				// Have arrived here from another part of the checkout process
				// Restore the data previously entered by the user
				if ($this->_file_exists_in_include_path('Crypt/Blowfish.php') &&
						$this->_file_exists_in_include_path('PEAR.php')) {
					// The PEAR Crypt Blowfish package can be used, use it to decrypt the Credit
					// Card Details. See pre_confirmation_check for encryption information.
					require_once('Crypt/Blowfish.php');
					
					$bf = new Crypt_Blowfish(
						substr(MODULE_PAYMENT_CARDSAVE_ENCRYPTION_KEYPHRASE, 0, 56));
					
					$plaintext = $bf->decrypt($_SESSION['cardsave_data_entered']);
					
					$data_entered = unserialize(base64_decode($plaintext));
				} else {
					// Card Details were stored unencrypted in the session...
					// COULD BE A SECURITY RISK, it is HIGHLY ADVISED that the PEAR Crypt Blowfish
					// Package is installed!
					$data_entered = unserialize(base64_decode($_SESSION['cardsave_data_entered']));
				}
				
				$cardsave_card_owner = $data_entered['cardsave_card_owner'];
				$cardsave_card_type = $data_entered['cardsave_card_type'];
				$cardsave_card_number = $data_entered['cardsave_card_number'];
				$cardsave_card_cv2 = $data_entered['cardsave_card_cv2'];
				$cardsave_card_expires_month = $data_entered['cardsave_card_expires_month'];
				$cardsave_card_expires_year = $data_entered['cardsave_card_expires_year'];
				
				if (isset($data_entered['cardsave_card_start_month'])) {
					$cardsave_card_start_month = $data_entered['cardsave_card_start_month'];
					$cardsave_card_start_year = $data_entered['cardsave_card_start_year'];
				}
				
				if (isset($data_entered['cardsave_card_issue'])) {
					$cardsave_card_issue = $data_entered['cardsave_card_issue'];
				}
				
				if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes' && 
						isset($data_entered['cardsave_use_test_case'])) {
					$cardsave_use_test_case = $data_entered['cardsave_use_test_case'];
				}
			}
		}
		
		$selection = array(
			'id' => $this->code,
			'module' => MODULE_PAYMENT_CARDSAVE_TEXT_CATALOG_TITLE
			);
		
		// Display icons for the list of cards accepted?
		if (MODULE_PAYMENT_CARDSAVE_SHOW_CARDS_ACCEPTED == 'Yes') {
			// Build the list of cards accepted
			$cards_accepted_images_source = '';
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/visa.png', MODULE_PAYMENT_CARDSAVE_TEXT_VISA, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_MASTERCARD == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/mc.png', MODULE_PAYMENT_CARDSAVE_TEXT_MASTERCARD, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_DEBIT == 'Yes') {
        echo DIR_WS_TEMPLATE_IMAGES;
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES . 'card_icons/visa_debit.png', MODULE_PAYMENT_CARDSAVE_TEXT_VISA_DEBIT, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_SOLO == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/solo.png', MODULE_PAYMENT_CARDSAVE_TEXT_SOLO, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_MAESTRO == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/maestro.png', MODULE_PAYMENT_CARDSAVE_TEXT_MAESTRO, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_ELECTRON == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/visa_electron.png', MODULE_PAYMENT_CARDSAVE_TEXT_VISA_ELECTRON, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA != 'Yes' && MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_PURCHASING == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/visa.png', MODULE_PAYMENT_CARDSAVE_TEXT_VISA_PURCHASING, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_AMERICAN_EXPRESS == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/amex.png', MODULE_PAYMENT_CARDSAVE_TEXT_AMERICAN_EXPRESS, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_DINERS_CLUB == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/dc.png', MODULE_PAYMENT_CARDSAVE_TEXT_DINERS_CLUB, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_JCB == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/jcb.png', MODULE_PAYMENT_CARDSAVE_TEXT_JCB, '', '', 'class="CardSaveCardIcon"');
			}
			if (MODULE_PAYMENT_CARDSAVE_ACCEPT_LASER == 'Yes') {
				$cards_accepted_images_source .= tep_image(DIR_WS_TEMPLATE_IMAGES  . 'card_icons/laser.png', MODULE_PAYMENT_CARDSAVE_TEXT_LASER, '', '', 'class="CardSaveCardIcon"');
			}
			$selection['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARDS_ACCEPTED,
				'field' => $cards_accepted_images_source
				);
		}
		
		// Build select gadget with list of test details which can be used to test the module
		if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes') {
			// Load the CardSave test cases
			require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/test_cases.php');
			
			// Output message to let user know what this field is for
			$selection['fields'][] = array(
					'title' => '',
					'field' => MODULE_PAYMENT_CARDSAVE_TEST_MODE_MESSAGE
					);
			
			$selection['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_TEST_CASE,
				'field' => tep_draw_pull_down_menu('cardsave_use_test_case', $test_cases, $cardsave_use_test_case)
				);
		}
		
		$selection['fields'][] = array(
			'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_OWNER,
			'field' => tep_draw_input_field('cardsave_card_owner', $cardsave_card_owner)
			);
		
		// Display any custom message specified in the admin
		if (MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS == 'Yes'
				&& MODULE_PAYMENT_CARDSAVE_ENABLE_CUSTOM_SURCHARGES_DISCOUNTS_MESSAGE == 'Yes'
				&& defined('MODULE_PAYMENT_CARDSAVE_CUSTOM_SURCHARGES_DISCOUNTS_MESSAGE')
				&& strlen(MODULE_PAYMENT_CARDSAVE_CUSTOM_SURCHARGES_DISCOUNTS_MESSAGE) > 0) {
			
			$selection['fields'][] = array(
				'title' => '',
				'field' => MODULE_PAYMENT_CARDSAVE_CUSTOM_SURCHARGES_DISCOUNTS_MESSAGE
				);
		}
		
		$selection['fields'][] = array(
			'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_TYPE,
			'field' => tep_draw_pull_down_menu('cardsave_card_type', $card_type, $cardsave_card_type)
			);
		
		if (MODULE_PAYMENT_CARDSAVE_DISABLE_CARD_NUMBER_AUTOCOMPLETE == 'Yes') {
			$selection['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_NUMBER,
				'field' => tep_draw_input_field('cardsave_card_number', $cardsave_card_number, 'autocomplete="off"')
				);
		} else {
			$selection['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_NUMBER,
				'field' => tep_draw_input_field('cardsave_card_number', $cardsave_card_number)
				);
			
		}
		
		$selection['fields'][] = array(
			'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_EXPIRES,
			'field' => tep_draw_pull_down_menu('cardsave_card_expires_month', $expires_month, $cardsave_card_expires_month) . '&nbsp;' . tep_draw_pull_down_menu('cardsave_card_expires_year', $expires_year, $cardsave_card_expires_year)
			);
		
		
		if (MODULE_PAYMENT_CARDSAVE_USE_CV2 == 'Yes') {
			if (MODULE_PAYMENT_CARDSAVE_DISABLE_CV2_AUTOCOMPLETE == 'Yes') {
				$selection['fields'][] = array(
					'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CV2,
					'field' => tep_draw_input_field('cardsave_card_cv2', $cardsave_card_cv2, 'size="4" maxlength="4" autocomplete="off"') . $cvv_test_mode_message
					);
			} else {
				$selection['fields'][] = array(
					'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CV2,
					'field' => tep_draw_input_field('cardsave_card_cv2', $cardsave_card_cv2, 'size="4" maxlength="4"') . $cvv_test_mode_message
					);
				
			}
		}
		
		if ($this->_useStartDate()) {
			$selection['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_START_DATE,
				'field' => tep_draw_pull_down_menu('cardsave_card_start_month', $start_month, $cardsave_card_start_month) . '&nbsp;' . tep_draw_pull_down_menu('cardsave_card_start_year', $start_year, $cardsave_card_start_year)
				);
		}
		if ($this->_useIssueNumber()) {
			$selection['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_ISSUE,
				'field' => tep_draw_input_field('cardsave_card_issue', $cardsave_card_issue, 'size="2" maxlength="2"')
				);
		}
		
		return $selection;
	}
	
	// }}}
	
	
	// {{{ pre_confirmation_check()

	/**
	 * Evaluates the Credit/Debit Card Type for acceptance and the validity of the Card Number &
	 * Expiration Date. Redirects back to Card Details entry page if invalid data detected.
	 *
	 * @access  public
	 * @param   none
	 * @return  none
	 */
	function pre_confirmation_check()
	{
		// Store the data entered so far so that user is not required to enter everything again if
		// anything is wrong or if they come back to the payment page to change some detail(s) /////
		$data_entered = array();
		
		$data_entered['cardsave_card_owner'] = $_POST['cardsave_card_owner'];
		$data_entered['cardsave_card_type'] = $_POST['cardsave_card_type'];
		$data_entered['cardsave_card_number'] = $_POST['cardsave_card_number'];
		
		$data_entered['cardsave_card_expires_month'] = $_POST['cardsave_card_expires_month'];
		$data_entered['cardsave_card_expires_year'] = $_POST['cardsave_card_expires_year'];
		
		if (isset($_POST['cardsave_card_cv2'])) {
			$data_entered['cardsave_card_cv2'] = $_POST['cardsave_card_cv2'];
		} else {
			$data_entered['cardsave_card_cv2'] = '';
		}
		
		if (isset($_POST['cardsave_card_start_year'])) {
			$data_entered['cardsave_card_start_month'] = $_POST['cardsave_card_start_month'];
			$data_entered['cardsave_card_start_year'] = $_POST['cardsave_card_start_year'];
		}
		
		if (isset($_POST['cardsave_card_issue'])) {
			$data_entered['cardsave_card_issue'] = $_POST['cardsave_card_issue'];
		}
		
		if (isset($_POST['cardsave_use_test_case'])) {
			$data_entered['cardsave_use_test_case'] = $_POST['cardsave_use_test_case'];
		}
		
		if (MODULE_PAYMENT_CARDSAVE_STORE_DETAILS_IN_SESSION == 'Yes') {
			// Data entered is stored in the session as an base64 encoded, serialised array, with
			// optional encryption. However it is HIGHLY RECOMMENDED that encryption is used as it
			// prevents other users on your server from possibly obtaining Card Details from
			// the session file. As far as we are aware it is illegal to disregard this possibility
			// but we can take no responsibility for this information.. YOU MUST CHECK THIS OUT
			// YOURSELF!
			$plaintext = base64_encode(serialize($data_entered));
			
			if ($this->_file_exists_in_include_path('Crypt/Blowfish.php') &&
					$this->_file_exists_in_include_path('PEAR.php')) {
				// The PEAR Crypt Blowfish package can be used, use it to encrypt the  Card Details.
				// This should provide reliable security for the protection of  Card Details
				// stored within the session, especially given that the session is a temporary
				// entity which expires when the user logs out or doesn't use the site for a
				// certain period of time.
				require_once('Crypt/Blowfish.php');
				
				$bf = new Crypt_Blowfish(
					substr(MODULE_PAYMENT_CARDSAVE_ENCRYPTION_KEYPHRASE, 0, 56));
				
				$encrypted = $bf->encrypt($plaintext);
				
				$_SESSION['cardsave_data_entered'] = $encrypted;
			} else {
				// Card Details are being stored unencrypted in the session...
				// COULD BE A SECURITY RISK, it is HIGHLY ADVISED that the PEAR Crypt Blowfish
				// Package is installed and used! See above!
				$_SESSION['cardsave_data_entered'] = $plaintext;
			}
		}
		
		if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes' && $_POST['cardsave_use_test_case'] != '-1') {
			// User has selected a test case to be used
			
			// Load the CardSave test cases
			require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/test_cases.php');
			
			$this->card_owner = $test_case_info[$_POST['cardsave_use_test_case']]['card_owner'];
			$this->card_type = $test_case_info[$_POST['cardsave_use_test_case']]['card_type'];
			$this->card_number = $test_case_info[$_POST['cardsave_use_test_case']]['card_number'];
			$this->card_expires_month = $test_case_info[$_POST['cardsave_use_test_case']]['card_expires_month'];
			$this->card_expires_year = $test_case_info[$_POST['cardsave_use_test_case']]['card_expires_year'];
			
			if (MODULE_PAYMENT_CARDSAVE_USE_CV2 != 'Yes' ) {
				$this->card_cv2 = '000';
			} else {
				$this->card_cv2 = $test_case_info[$_POST['cardsave_use_test_case']]['card_cv2'];
			}
			
			$this->card_start_month = $test_case_info[$_POST['cardsave_use_test_case']]['card_start_month'];
			$this->card_start_year = $test_case_info[$_POST['cardsave_use_test_case']]['card_start_year'];
			
			if ($this->_useIssueNumber()) {
				$this->card_issue = '';
			}
			
			$this->use_test_case = $_POST['cardsave_use_test_case'];
		} else {
			// Use the details entered by the user
			include(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.cardsaveCardValidation.php');
			
			$cardsave_card_validation = new cardsaveCardValidation();
			$result = $cardsave_card_validation->validate($_POST['cardsave_card_number'],
				$_POST['cardsave_card_expires_month'], $_POST['cardsave_card_expires_year']);
			$error = '';
			switch ($result) {
				case -1:
					$error = MODULE_PAYMENT_CARDSAVE_TEXT_CARD_UNKNOWN_ERROR;
					break;
				case -2:
				case -3:
				case -4:
					$error = MODULE_PAYMENT_CARDSAVE_TEXT_CARD_EXPIRY_ERROR;
					break;
				case false:
					$error = MODULE_PAYMENT_CARDSAVE_TEXT_CARD_NUMBER_ERROR;
					break;
			}
			
			if ($_POST['cardsave_card_type'] == 'xxx') {
				// Type of card not selected!
				$result = false;
				$error = MODULE_PAYMENT_CARDSAVE_TEXT_CARD_TYPE_ERROR;
			} else {
				// Check the card type and warn if it doesn't match the card number entered
				
				// Load the CardSave files
				require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/Common.php');
				require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/PaymentSystem.php');
				require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/SOAP.php');
				
				// Set up Gateway Account details providing Merchant ID and Password
				$mdMerchantDetails = new MerchantDetails(MODULE_PAYMENT_CARDSAVE_GATEWAY_MERCHANT_ID,
					MODULE_PAYMENT_CARDSAVE_GATEWAY_PASSWORD);
				
				// Set up Pass output data
				$szPassOutputData = 'Integration Module By Ceon';
				
				// Output objects
				$goGatewayOutput = null;
				$tomTransactionOutputMessage = null;
				
				$rgeplRequestGatewayEntryPointList = $this->_getGatewayEntryPointList();
				
				$getCardType = new getCardType($rgeplRequestGatewayEntryPointList, 1, null,
					$mdMerchantDetails, $data_entered['cardsave_card_number'], $szPassOutputData);
				$boTransactionProcessed = $getCardType->processTransaction($goGatewayOutput,
					$tomTransactionOutputMessage);
				
				if (is_null($goGatewayOutput)) {
					// Transaction didn't run
					
					// Redirect back to payment page and display error message
					$error = $this->_encodeErrorMessage(
						MODULE_PAYMENT_CARDSAVE_TEXT_TRANSACTION_PROBLEM);
					$payment_error_return = 'payment_error=' . $this->code . '&error=' .
						urlencode($error);
					
					tep_redirect(tep_href_link(FILENAME_CHECKOUT_PAYMENT, $payment_error_return,
						'SSL', true, false));
				} else {
					// Contacted CardSave successfully!
					
					// Parse CardSave response object
					$cardsave_return_values = $this->_parseCardSaveCardTypeResponse($goGatewayOutput,
						$tomTransactionOutputMessage);
					
					if ($cardsave_return_values['status_code'] == '30') {
						$result = false;
						$error = $this->_identifyErrorMessage($cardsave_return_values['status_code'],
							$cardsave_return_values['message'], $goGatewayOutput,
							$tomTransactionOutputMessage);
					} else if (isset($cardsave_return_values['card_type']) &&
							$cardsave_return_values['card_type'] != $_POST['cardsave_card_type']) {
						$result = false;
						$error = sprintf(MODULE_PAYMENT_CARDSAVE_TEXT_CARD_TYPE_DOES_NOT_MATCH,
							$this->_getCardTypeNameForCode($cardsave_return_values['card_type']));
					}
				}
			}
			
			if (($result == false) || ($result < 1)) {
				// The user has not entered valid Card Details, redirect back to the input form
				
				// Encode the error message and redirect
				$error = $this->_encodeErrorMessage($error);
				$payment_error_return = 'payment_error=' . $this->code . '&error=' .
					urlencode($error);
				
				tep_redirect(tep_href_link(FILENAME_CHECKOUT_PAYMENT, $payment_error_return, 'SSL',
					true, false));
			}
			
			// Card seems to be valid, store the details found
			$this->card_owner = $_POST['cardsave_card_owner'];
			$this->card_type = $_POST['cardsave_card_type'];
			$this->card_number = $cardsave_card_validation->card_number;
			$this->card_expires_month = $cardsave_card_validation->card_expiry_month;
			$this->card_expires_year = $cardsave_card_validation->card_expiry_year;
			
			if (MODULE_PAYMENT_CARDSAVE_USE_CV2 != 'Yes' ) {
				$this->card_cv2 = '000';
			} else {
				$this->card_cv2 = $_POST['cardsave_card_cv2'];
			}
			
			if ($this->_useStartDate()) {
				$this->card_start_month = $_POST['cardsave_card_start_month'];
				$this->card_start_year = $_POST['cardsave_card_start_year'];
			}
			
			if ($this->_useIssueNumber()) {
				$this->card_issue = $_POST['cardsave_card_issue'];
			}
			
			$this->use_test_case = $_POST['cardsave_use_test_case'];
		}
		
		// Now that a card type has been selected, must check if the Surcharges/Discounts Order
		// Total module is in use and, if so, store the possible titles which this card type can
		// have displayed (either a surcharge title or discount title)
		if (MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS == 'Yes' &&
				isset($GLOBALS['ot_payment_surcharges_discounts'])) {
			
			$table_of_rates = $this->_getSurchargeDiscountTableOfRates($this->card_type);
			
			if ($table_of_rates !== false) {
				$_SESSION['payment_surcharge_discount']['table_of_rates'] = $table_of_rates;
				$_SESSION['payment_surcharge_discount']['title'] = '';
				
				// Check if some informational text has been specified for this card type (and
				// language)
				$surcharge_discount_long_text = trim(
					constant('MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_' .
					$this->card_type . '_LONG_' . $_SESSION['languages_id']));
				if (strlen($surcharge_discount_long_text) > 0) {
					$_SESSION['payment_surcharge_discount']['title'] =
						$surcharge_discount_long_text;
				}
				
				if ($_SESSION['payment_surcharge_discount']['title'] == '') {
					// Use default text to inform the user about the surcharge/discount
					$_SESSION['payment_surcharge_discount']['default_title_surcharge'] =
						MODULE_PAYMENT_CARDSAVE_TEXT_SURCHARGE_LONG;
					$_SESSION['payment_surcharge_discount']['default_title_discount'] =
						MODULE_PAYMENT_CARDSAVE_TEXT_DISCOUNT_LONG;
				}
			} else {
				if (isset($_SESSION['payment_surcharge_discount'])) {
					unset($_SESSION['payment_surcharge_discount']);
				}
			}
		}
	}
	
	// }}}
	
	
	// {{{ confirmation()

	/**
	 * Builds the Card Information for display on the Checkout Confirmation Page
	 *
	 * @access  public
	 * @param   none
	 * @return  array  The list of Field Titles and their associated Values.
	 *
	 *                 Format:
	 *
	 *                 fields  array  The list of field titles and values stored as hashes.
	 */
	function confirmation()
	{
		// To reorder the output just adjust the order that the fields are added to the array
		// E.g. Below, "Expires" can be reordered below "Start" by moving its section appropriately
		
		if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes' && $this->use_test_case != '-1') {
			// Load the CardSave test cases
			require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/test_cases.php');
			
			$confirmation['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_TEST_CASE,
				'field' => $test_cases[$this->use_test_case]['text']
				);
		}
		
		// Get the name for the card type selected as defined in the language definitions file
		$card_type_name = $this->_getCardTypeNameForCode($this->card_type);
		
		$confirmation['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_TYPE,
				'field' => $card_type_name
			);
		$confirmation['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_OWNER,
				'field' => $this->card_owner
			);
		$confirmation['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_NUMBER,
				'field' => substr($this->card_number, 0, 4) .
					str_repeat('X', (strlen($this->card_number) - 8)) .
					substr($this->card_number, -4)
			);
		$confirmation['fields'][] = array(
			'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_EXPIRES,
			'field' => strftime('%B, %Y', mktime(0, 0, 0, $this->card_expires_month, 1,
				$this->card_expires_year))
			);
		if (MODULE_PAYMENT_CARDSAVE_USE_CV2 == 'Yes') {
			$confirmation['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CV2,
				'field' => $this->card_cv2
				);
		}
		if ($this->_useStartDate() && $this->card_start_year != '') {
			$confirmation['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_START_DATE,
				'field' => strftime('%B, %Y', mktime(0, 0, 0, $this->card_start_month, 1,
					'20' . $this->card_start_year))
				);
		}
		if ($this->_useIssueNumber() && $this->card_issue != '') {
			$confirmation['fields'][] = array(
				'title' => MODULE_PAYMENT_CARDSAVE_TEXT_CARD_ISSUE,
				'field' => $this->card_issue
				);
		}
		
		return $confirmation;
	}
	
	// }}}
	
	
	// {{{ process_button()

	/**
	 * Builds a list of the Card Details for this transaction, each piece of data being
	 * stored as a hidden HTML form field.
	 *
	 * @access  public
	 * @param   none
	 * @return  string  The Card Details for this transaction as a list of hidden form fields
	 */
	function process_button()
	{
		// These are hidden fields on the checkout confirmation page
		$process_button_string = tep_draw_hidden_field('card_owner', $this->card_owner) .
			tep_draw_hidden_field('card_expires', $this->card_expires_month .
			substr($this->card_expires_year, -2)) .
			tep_draw_hidden_field('card_type', $this->card_type) .
			tep_draw_hidden_field('card_number', $this->card_number);
		
		if (MODULE_PAYMENT_CARDSAVE_USE_CV2 == 'Yes') {
			$process_button_string .= tep_draw_hidden_field('card_cv2', $this->card_cv2);
		}
		
		if ($this->_useStartDate()) {
			$process_button_string .= tep_draw_hidden_field('card_start', $this->card_start_month .
				substr($this->card_start_year, -2));
		}
		
		if ($this->_useIssueNumber()) {
			$process_button_string .= tep_draw_hidden_field('card_issue', $this->card_issue);
		}
		
		if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes') {
			$process_button_string .= tep_draw_hidden_field('use_test_case', $this->use_test_case);
		}
		
		$process_button_string .= tep_draw_hidden_field(tep_session_name(), tep_session_id());
		
		return $process_button_string;
	}
	
	// }}}
	
	
	// {{{ before_process()

	/**
	 * The main guts of the payment module as it were, this method formats the data appropriately
	 * for sending to the payment gateway, actually sends the data and then processes the response.
	 * If a problem occurs this will redirect back to the Card Details entry page.
	 *
	 * In the case of a 3D Secure transaction, this method will receive the callback reponse and
	 * process it appropriately.
	 *
	 * @access  public
	 * @param   none
	 * @return  none
	 */
	function before_process()
	{
		global $db, $order, $currencies;
		
		// Load the CardSave files
		require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/Common.php');
		require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/PaymentSystem.php');
		require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/SOAP.php');
		
		// Check if this method has been called as a result of a callback from a 3D Secure
		// verification
		if (!isset($_POST['MD']) || !isset($_POST['PaRes'])) {
			// Method has been called normally, submit card details to CardSave /////////////////////
			
			// Store the card details for this order
			// Hide middle digits for the card number
			$order->info['cc_number'] = substr($_POST['card_number'], 0, 4) .
				str_repeat('X', (strlen($_POST['card_number']) - 8)) .
				substr($_POST['card_number'], -4);
			
			$order->info['cc_expires'] = $_POST['card_expires'];
			$order->info['cc_type'] =
				substr($this->_getCardTypeNameForCode($_POST['card_type'], false), 0, 20);
			$order->info['cc_owner'] = $_POST['card_owner'];
			
			if ($this->_useStartDate()) {
				$order->info['cc_start'] = $_POST['card_start'];
			}
			if ($this->_useIssueNumber()) {
				$order->info['cc_issue'] = $_POST['card_issue'];
			}
			
			// DATA PREPARATION ////////////////////////////////////////////////////////////////////
			
			// Transaction Type
			switch (MODULE_PAYMENT_CARDSAVE_TRANSACTION_TYPE) {
				case "Immediate Charge":
					$transaction_type = CardSaveTransactionType::SALE;
					break;
				case "Pre-authorised":
					$transaction_type = CardSaveTransactionType::PREAUTH;
					break;
				case "Stored":
					$transaction_type = CardSaveTransactionType::STORE;
					break;
			}
			
			// AVS Settings
			$avs_policy = '';
			switch (MODULE_PAYMENT_CARDSAVE_AVS_POLICY) {
				case "Both checks CANNOT fail.":
					$avs_policy .= 'E';
					break;
				case "Address numeric/postcode check CAN fail but NOT BOTH.":
					$avs_policy .= 'B';
					break;
				case "Only address numeric check CAN fail(Default)":
					$avs_policy .= 'P';
					break;
				case "Only postcode check CAN fail":
					$avs_policy .= 'A';
					break;
				case "Address numeric/postcode check CAN fail.":
					$avs_policy .= 'N';
					break;
			}
      
			switch (MODULE_PAYMENT_CARDSAVE_AVS_POLICY_ADDRESS_NUMERIC_PARTIAL_MATCH) {
				case "Partial matches should be treated as a fail. (Default)":
					$avs_policy .= 'F';
					break;
				case "Partial matches should be treated as a pass.":
					$avs_policy .= 'P';
					break;
			}
            
			switch (MODULE_PAYMENT_CARDSAVE_AVS_POLICY_POSTCODE_PARTIAL_MATCH) {
				case "Partial matches should be treated as a fail. (Default)":
					$avs_policy .= 'F';
					break;
				case "Partial matches should be treated as a pass.":
					$avs_policy .= 'P';
					break;
			}
			switch (MODULE_PAYMENT_CARDSAVE_AVS_POLICY_RESULTS_UNKNOWN) {
				case "Both checks having unknown results should be treated as a fail. (Default)":
					$avs_policy .= 'F';
					break;
				case "Both checks have unknown results should be treated as a pass.":
					$avs_policy .= 'P';
					break;
			}
			
			// CV2 Settings
			if (MODULE_PAYMENT_CARDSAVE_USE_CV2 == 'Yes') {
				$cv2_policy = '';
				switch (MODULE_PAYMENT_CARDSAVE_CV2_POLICY) {
					case "The CV2 check must not fail. (Default)":
						$cv2_policy .= 'F';
						break;
					case "The CV2 check can fail.":
						$cv2_policy .= 'P';
						break;
				}
				switch (MODULE_PAYMENT_CARDSAVE_CV2_POLICY_RESULTS_UNKNOWN) {
					case "The CV2 check having unknown results should be treated as a fail. (Default)":
						$cv2_policy .= 'F';
						break;
					case "The CV2 check having unknown results should be treated as a pass.":
						$cv2_policy .= 'P';
						break;
				}
			}
			
			// 3D-Secure Settings
			switch (MODULE_PAYMENT_CARDSAVE_3D_SECURE_POLICY) {
				case "Use 3D-Secure whenever possible. (Default)":
					$apply_3d_secure = true;
					break;
				case "Only use 3D-Secure for card types that absolutely require it.":
					$apply_3d_secure = false;
					break;
			}
			
			// Determine if the amount to be charged should be charged in the currently selected
			// currency or the Shop Prices' Base Currency
			$currency_amount = 0;
			$currency_code = '';
			
			// Get the current currency in which the customer is viewing prices
			$current_view_currency = $_SESSION['currency'];
			
			// Check if this currency can be accepted by the Shop
			$accepted_currencies_for_transactions = explode(',',
				MODULE_PAYMENT_CARDSAVE_ACCEPTED_CURRENCIES);
			if (in_array($current_view_currency, $accepted_currencies_for_transactions)) {
				// The current currency has an associated Merchant Account and so can be used
				$currency_code = $current_view_currency;
				
				$currency_amount = number_format(($order->info['total']) *
					$currencies->get_value($currency_code), 2, '.', '');
			}
			
			if ($currency_code == '') {
				// Must carry out the transaction using the Default Merchant Account's currency
				$currency_code = MODULE_PAYMENT_CARDSAVE_DEFAULT_CURRENCY;
				
				// Check if the currency used to enter all products prices (the shop's default
				// currency) matches the currency used by the Default Merchant Account
				if (DEFAULT_CURRENCY == $currency_code) {
					$currency_amount = number_format(($order->info['total']), 2, '.', '');
				} else {
					// The prices in the shop have not been entered in a currency for which a
					// merchant account exists. Must convert the order total to the currency used by
					// the default merchant account
					$currency_amount = number_format(($order->info['total']) *
						$currencies->get_value($currency_code), 2, '.', '');
				}
			}
			
			// Convert currency from 2 point decimal to minor currency
			$currency_amount = $currency_amount * 100;
			
			// Unique Transaction ID, max of 40 characters.  Format:  Time - Customer ID - Customer
			// Last Name. (Must make sure no illegal characters present in last name first!)
			$unique_transaction_id = substr(time() . '-' .
				$this->_cleanString($_SESSION['customer_id']) . '-' .
				$this->_cleanString($order->customer['lastname']), 0, 40);
			
			// Set up Gateway Account details providing Merchant ID and Password
			$mdMerchantDetails = new MerchantDetails(MODULE_PAYMENT_CARDSAVE_GATEWAY_MERCHANT_ID,
				MODULE_PAYMENT_CARDSAVE_GATEWAY_PASSWORD);
			
			// Set up Message details
			$ttTransactionType = new NullableTRANSACTION_TYPE($transaction_type);
			$mdMessageDetails = new MessageDetails($ttTransactionType);
			
			// Set up Transaction Control
			$boEchoCardType = new NullableBool(true);
			$boEchoAmountReceived = new NullableBool(true);
			$boEchoAVSCheckResult = new NullableBool(true);
			$boEchoCV2CheckResult = new NullableBool(true);
			$nDuplicateDelay = new NullableInt(60);
			$szAVSOverridePolicy = $avs_policy;
			$szCV2OverridePolicy = $cv2_policy;
			$boThreeDSecureOverridePolicy = new NullableBool($apply_3d_secure);
			$tcTransactionControl = new TransactionControl($boEchoCardType, $boEchoAVSCheckResult,
				$boEchoCV2CheckResult, $boEchoAmountReceived, $nDuplicateDelay,
				$szAVSOverridePolicy, $szCV2OverridePolicy, $boThreeDSecureOverridePolicy, '', null,
				null);
			
			// Set up Transaction Details
			$nAmount = new NullableInt($currency_amount);
			
			// Convert standard 3-character currency code into CardSave numerical code
			$cardsave_currency_code = null;
			switch ($currency_code) {
				case 'GBP':
					$cardsave_currency_code = 826;
					break;
				case 'USD':
					$cardsave_currency_code = 840;
					break;
				case 'EUR':
					$cardsave_currency_code = 976;
					break;
				case 'AUD':
					$cardsave_currency_code = 036;
					break;
				case 'CAD':
					$cardsave_currency_code = 124;
					break;
			}
			if (is_null($cardsave_currency_code)) {
				// Redirect back to payment page and display error message
				$error = $this->_encodeErrorMessage(sprintf(
					MODULE_PAYMENT_CARDSAVE_TEXT_UNSUPPORTED_CURRENCY_CODE, $currency_code));
				$payment_error_return = 'payment_error=' . $this->code . '&error=' .
					urlencode($error);
				
				tep_redirect(tep_href_link(FILENAME_CHECKOUT_PAYMENT, $payment_error_return, 'SSL',
					true, false));
			}
			$nCurrencyCode = new NullableInt($cardsave_currency_code);
			$tdTransactionDetails = new TransactionDetails($mdMessageDetails, $nAmount,
				$nCurrencyCode, $unique_transaction_id, substr(STORE_NAME .
				' - osCommerce Cart Order @ ' . date('Y-m-d H:i:s', time()), 0, 256),
				$tcTransactionControl, NULL);
			
			// Set up Card details
			$nExpiryDateMonth = new NullableInt(substr($_POST['card_expires'], 0, 2));
			$nExpiryDateYear = new NullableInt(substr($_POST['card_expires'], 2, 2));
			$ccdExpiryDate = new CreditCardDate($nExpiryDateMonth, $nExpiryDateYear);
			if ($this->_useStartDate() && strlen($_POST['card_start']) == 4) {
				$nStartDateMonth = new NullableInt(substr($_POST['card_start'], 0, 2));
				$nStartDateYear = new NullableInt(substr($_POST['card_start'], 2, 2));
			} else {
				$nStartDateMonth = new NullableInt(null);
				$nStartDateYear = new NullableInt(null);
			}
			$ccdStartDate = new CreditCardDate($nStartDateMonth, $nStartDateYear);
			if ($this->_useIssueNumber() && strlen(trim($_POST['card_issue'])) > 0) {
				$nIssueNumber = $_POST['card_issue'];
			} else {
				$nIssueNumber = null;
			}
			$nCV2 = isset($_POST['card_cv2']) ? $_POST['card_cv2'] : null;
			$cdCardDetails = new CardDetails($_POST['card_owner'], $_POST['card_number'],
				$ccdExpiryDate, $ccdStartDate, $nIssueNumber, $nCV2);
			
			
			// Set up Customer Details
			if (MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED == 'Yes' &&
					$_POST['use_test_case'] != '-1') {
				// Use selected test address
				
				// Load the CardSave test cases
				require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/test_cases.php');
				
				$nCountryCode = new NullableInt(826); // Test numbers use UK
				$adBillingAddress = new AddressDetails(substr(
					$test_case_info[$_POST['use_test_case']]['card_address_numeric'], 0, 100),
					null, null, null,
					substr($test_case_info[$_POST['use_test_case']]['card_city'], 0, 50),
					substr($test_case_info[$_POST['use_test_case']]['card_state'], 0, 50),
					substr($test_case_info[$_POST['use_test_case']]['card_postcode'], 0, 50),
					$nCountryCode);
			} else {
				// Convert Zen Cart country code into CardSave country code
				require(DIR_FS_CATALOG . DIR_WS_MODULES . 'payment/cardsave/ISOCountries.php');
				
				$country_code = null;
				
				$tep_country_code = $order->billing['country']['iso_code_3'];
				
				for ($country_i = 0; $country_i < $iclISOCountryList->getCount() - 1; $country_i++) {
					if ($iclISOCountryList->getAt($country_i)->getCountryNameShort() == $tep_country_code) {
						$country_code = $iclISOCountryList->getAt($country_i)->getISOCode();
						break;
					}
				}
				
				if (is_null($country_code)) {
					
				}
				
				$nCountryCode = new NullableInt($country_code);
				$adBillingAddress = new AddressDetails(
					substr($order->billing['street_address'], 0, 100),
					substr($order->billing['suburb'], 0, 50), null, null,
					substr($order->billing['city'],0, 50),
					substr($order->billing['state'], 0, 50),
					substr($order->billing['postcode'], 0, 50), $nCountryCode);
			}
			
			$cdCustomerDetails = new CustomerDetails($adBillingAddress,
				substr($order->customer['email_address'], 0, 100),
				substr($order->customer['telephone'], 0, 50), $_SERVER['REMOTE_ADDR']);
			
			// Set up Pass output data
			$szPassOutputData = 'Integration Module By Ceon';
			
			// RECORD TRANSACTION //////////////////////////////////////////////////////////////////
			// Record the details for this transaction in case something goes wrong and the customer
			// isn't returned by cardsave correctly after it has processed the transaction... allows
			// for the ability to warn of, and prevent, duplicate transactions
			
			// (TO BE ADDED) ///////
			
			// SEND DATA ///////////////////////////////////////////////////////////////////////////
			// Post order info data to CardSave. (Will fail if curl is not installed)
			
			// Output objects
			$goGatewayOutput = null;
			$tomTransactionOutputMessage = null;
			
			$rgeplRequestGatewayEntryPointList = $this->_getGatewayEntryPointList();
			
			$cdtCardDetailsTransaction = new CardDetailsTransaction(
				$rgeplRequestGatewayEntryPointList, 1, null, $mdMerchantDetails,
				$tdTransactionDetails, $cdCardDetails, $cdCustomerDetails, $szPassOutputData);
			$boTransactionSuccessful = $cdtCardDetailsTransaction->ProcessTransaction(
				$goGatewayOutput, $tomTransactionOutputMessage);
			
			if (is_null($goGatewayOutput)) {
				// Transaction didn't run
				if (MODULE_PAYMENT_CARDSAVE_DEBUGGING_ENABLED != 'Yes') {
					// Redirect back to payment page and display error message
					$error = $this->_encodeErrorMessage(
						MODULE_PAYMENT_CARDSAVE_TEXT_TRANSACTION_PROBLEM);
					$payment_error_return = 'payment_error=' . $this->code . '&error=' .
						urlencode($error);
					
					tep_redirect(tep_href_link(FILENAME_CHECKOUT_PAYMENT, $payment_error_return,
						'SSL', true, false));
				} else {
					// Debug mode is enabled so allow the debug information to be output later using
					// dummy values where appropriate
					$this->_cardsave_return_values = null;
					
					// Record temporary copy of returned object for debug purposes
					$unprocessed_cardsave_response = $goGatewayOutput;
				}
			} else {
				// Contacted CardSave successfully!
				
				// Record temporary copy of returned object for debug purposes
				$unprocessed_cardsave_response = $goGatewayOutput;
				
				// Parse CardSave response object and store returned values
				$this->_cardsave_return_values = $this->_parseCardSaveTransactionResponse(
					$goGatewayOutput, $tomTransactionOutputMessage);
			}
			
			// Debugging output
			if (MODULE_PAYMENT_CARDSAVE_DEBUGGING_ENABLED == 'Yes') {
				echo "<html><head><title>CardSave Debug Output</title></head><body>\n";
				echo "<pre>\n\n";
				echo "-------------------------------------\n";
				echo "Transaction response:\n";
				echo "-------------------------------------\n";
				echo var_dump($boTransactionSuccessful) . "\n";
				echo "-------------------------------------\n";
				echo "Unprocessed Gateway Output received:\n";
				echo "-------------------------------------\n";
				echo var_dump($unprocessed_cardsave_response) . "\n";
				echo "-------------------------------------\n";
				echo "Unprocessed Message received:\n";
				echo "-------------------------------------\n";
				echo var_dump($tomTransactionOutputMessage) . "\n";
				echo "------------------------------------------\n";
				echo "Data received from CardSave after processing:\n";
				echo "------------------------------------------\n";
				var_dump($this->_cardsave_return_values);
				echo "\n----------------------------\n";
				echo "Card Data sent to CardSave:\n";
				echo "----------------------------\n";
				echo var_dump($cdCardDetails);
				echo "\n----------------------------\n";
				echo "Transaction Data sent to CardSave:\n";
				echo "----------------------------\n";
				echo var_dump($tdTransactionDetails);
				echo "\n------------------------------\n";
				echo "Customer Data sent to CardSave:\n";
				echo "------------------------------\n";
				echo var_dump($cdCustomerDetails);
				echo "----------------------------\n\n";
				
				if (MODULE_PAYMENT_CARDSAVE_STORE_DETAILS_IN_SESSION == 'Yes') {
					if ($this->_file_exists_in_include_path('Crypt/Blowfish.php') &&
							$this->_file_exists_in_include_path('PEAR.php')) {
						echo "-----------------------------\n";
						echo "Blowfish Encryption was used.\n";
						echo "-----------------------------\n\n";
					} else {
						echo "--------------------\n";
						echo "Encryption NOT USED!\n";
						echo "--------------------\n\n";
					}
				} else {
					echo "----------------------------------------------------------------\n";
					echo "Details not being stored in session so encryption not necessary.\n";
					echo "----------------------------------------------------------------\n\n";
				}
				
				// Output the include path details
				echo "------------------------------\n";
				echo "Include paths for this system:\n";
				echo "------------------------------\n\n";
				echo get_include_path() . "\n\n";
				
				echo "----------------------\n";
				echo "Safe mode in use?: " . (ini_get('safe_mode') == 1 ? 'yes' : 'no') . "\n";
				echo "----------------------\n\n";
				
				echo "------------------------------------\n";
				echo "open_basedir restricted directories:\n";
				echo "------------------------------------\n\n";
				echo ini_get('open_basedir') . "\n\n";
				
				echo "\n-------------------\n";
				echo "Transaction status:\n";
				echo "-------------------\n";
				if ($this->_cardsave_return_values['status_code'] == 0) {
					echo "Transaction was successful!\n";
				} else if ($this->_cardsave_return_values['status_code'] == 3) {
					echo "3D Secure Authentication Required!\n";
				} else {
					echo "Transaction denied!\n";
					echo "Status: " . $this->_cardsave_return_values['status_code'] . "\n";
					echo "Message: " . $this->_cardsave_return_values['message'] . "\n\n";
					
					$error_message = $this->_identifyErrorMessage($this->_cardsave_return_values['status_code'], $this->_cardsave_return_values['message'], $goGatewayOutput, $tomTransactionOutputMessage);
					
					echo "Error Message: " . $error_message . "\n\n";
					
				}
				echo "\n\n</pre>\n</body>\n</html>";
				
				exit;
			}
			
			// Check what action must take place
			if ($this->_cardsave_return_values['status_code'] == 0 ) {
				// Transaction has gone through okay, let Zen Cart create the order etc.
				
			} else if ($this->_cardsave_return_values['status_code'] == 3) {
				// 3D-Secure authorisation is required
				// Must build message and form or auto-submitting form to redirect user to Bank's
				// website for entering of 3D-Secure authorisation details!
				
				// Store card details so they can be recorded against the order later
				$_SESSION['cardsave_order_card_details'] = array(
					'cc_number' => $order->info['cc_number'],
					'cc_expires' => $order->info['cc_expires'],
					'cc_type' => $order->info['cc_type'],
					'cc_owner' => $order->info['cc_owner']
					);
				
				if ($this->_useStartDate()) {
					$_SESSION['cardsave_order_card_details']['cc_start'] = $order->info['cc_start'];
				} else if (isset($_SESSION['cardsave_order_card_details']['cc_start'])) {
					unset($_SESSION['cardsave_order_card_details']['cc_start']);
				}
				if ($this->_useIssueNumber()) {
					$_SESSION['cardsave_order_card_details']['cc_issue'] = $order->info['cc_issue'];
				} else if (isset($_SESSION['cardsave_order_card_details']['cc_issue'])) {
					unset($_SESSION['cardsave_order_card_details']['cc_issue']);
				}
				
				// Redirect to the CardSave 3D-Secure output page/template
				$checkout_3d_secure_parameters = 'ACSURL=' . urlencode($this->_cardsave_return_values['acsurl']) .
					'&PaReq=' . urlencode($this->_cardsave_return_values['pareq']) .
					'&crossreference=' . urlencode($this->_cardsave_return_values['crossreference']);
					
				tep_redirect(tep_href_link(FILENAME_CHECKOUT_CARDSAVE_3D_SECURE, $checkout_3d_secure_parameters, 'SSL', true, false));
			} else {
				// Redirect back to the payment page with the appropriate error message
				$error_message = $this->_identifyErrorMessage($this->_cardsave_return_values['status_code'], $this->_cardsave_return_values['message'], $goGatewayOutput, $tomTransactionOutputMessage);
				
				$error_message = $this->_encodeErrorMessage($error_message);
				$payment_error_return = 'payment_error=' . $this->code . '&error=' . urlencode($error_message);
				tep_redirect(tep_href_link(FILENAME_CHECKOUT_PAYMENT, $payment_error_return, 'SSL', true, false));
			}
		} else {
			// Method has been called as a callback from a bank after the customer has entered
			// their 3D secure details, must forward these details on to CardSave ///////////////////
			
			// Restore the card details for the order so they can be recorded against it
			foreach ($_SESSION['cardsave_order_card_details'] as $key => $value) {
				$order->info[$key] = $value;
			}
			unset($_SESSION['cardsave_order_card_details']);
			
			// Set up 3D-Secure Request ////////////////////////////////////////////////////////////
			
			// Set up Gateway Account details providing Merchant ID and Password
			$mdMerchantDetails = new MerchantDetails(MODULE_PAYMENT_CARDSAVE_GATEWAY_MERCHANT_ID, MODULE_PAYMENT_CARDSAVE_GATEWAY_PASSWORD);
			
			$CrossReference = $_POST['MD'];
			$PaRES = $_POST['PaRes'];
			
			$tdsidThreeDSecureInputData = new ThreeDSecureInputData($CrossReference, $PaRES);
			
			// Set up Pass output data
			$szPassOutputData = 'Integration Module By Ceon';
			
			// Send 3D-Secure Request //////////////////////////////////////////////////////////////
			
			// Output objects
			$goGatewayOutput = null;
			$tomTransactionOutputMessage = null;
			
			$rgeplRequestGatewayEntryPointList = $this->_getGatewayEntryPointList();
			
			$tdsaThreeDSecureAuthentication = new ThreeDSecureAuthentication($rgeplRequestGatewayEntryPointList, 1, null, $mdMerchantDetails, $tdsidThreeDSecureInputData, $szPassOutputData);
			$boTransactionProcessed = $tdsaThreeDSecureAuthentication->processTransaction($goGatewayOutput, $tomTransactionOutputMessage);
			
			if (!$boTransactionProcessed) {
				// Transaction didn't run
				
				// Redirect back to payment page and display error message
				$error = $this->_encodeErrorMessage(MODULE_PAYMENT_CARDSAVE_TEXT_TRANSACTION_PROBLEM);
				$payment_error_return = 'payment_error=' . $this->code . '&error=' . urlencode($error);
				
				tep_redirect(tep_href_link(FILENAME_CHECKOUT_PAYMENT, $payment_error_return, 'SSL', true, false));
			} else {
				// Contacted CardSave successfully!
				
				// Parse CardSave response object and store returned values
				$this->_cardsave_return_values = $this->_parseCardSaveTransactionResponse($goGatewayOutput, $tomTransactionOutputMessage);
				
				// Check what action must take place
				if ($this->_cardsave_return_values['status_code'] == 0 ) {
					// Transaction has gone through okay, let Zen Cart create the order etc.
					return;
				} else {
					$error_message = $this->_identifyErrorMessage($this->_cardsave_return_values['status_code'], $this->_cardsave_return_values['message'], $goGatewayOutput, $tomTransactionOutputMessage);
				}
				
				// Redirect back to the payment page with the appropriate error message
				$error_message = $this->_encodeErrorMessage($error_message);
				$payment_error_return = 'payment_error=' . $this->code . '&error=' . urlencode($error_message);
				tep_redirect(tep_href_link(FILENAME_CHECKOUT_PAYMENT, $payment_error_return, 'SSL', true, false));
			}
		}
	}
	
	// }}}
	
	
	function after_process()
	{
		return false;
	}
	
	
	// {{{ _getGatewayEntryPointList()

	/**
	 * Gets the gateway entry point list for the module.
	 *
	 * @access  private
	 */
	function _getGatewayEntryPointList()
	{
		// Allow for a one-minute timeout for the attempt to connect to CardSave
		if (ini_get('safe_mode') != 1) {
			set_time_limit(70);
		} else {
			// Can't set a specific timeout, check what can be set!
			$max_timeout = ini_get('max_execution_time');
		}
		
		$PaymentProcessorFullDomain = MODULE_PAYMENT_CARDSAVE_GATEWAY_URL .
			(!is_null(MODULE_PAYMENT_CARDSAVE_GATEWAY_PORT && is_numeric(MODULE_PAYMENT_CARDSAVE_GATEWAY_PORT)) ?
			':' . MODULE_PAYMENT_CARDSAVE_GATEWAY_PORT : '');
		
		$rgeplRequestGatewayEntryPointList = new RequestGatewayEntryPointList();
		// The actual values to use for the entry points can be established in a number of ways
		// 1) By periodically issuing a call to GetGatewayEntryPoints
		// 2) By storing the values for the entry points returned with each transaction
		// 3) Speculatively firing transactions at https://gw1.xxx followed by gw2, gw3, gw4....
		// The lower the metric (2nd parameter) means that entry point will be attempted first,
		// EXCEPT if it is -1 - in this case that entry point will be skipped
		// NOTE: You do NOT have to add the entry points in any particular order - the list is sorted
		// by metric value before the transaction sumbitting process begins
		// The 3rd parameter is a retry attempt, so it is possible to try that entry point that number of times
		// before failing over onto the next entry point in the list
		$rgeplRequestGatewayEntryPointList->add("https://gw1.".$PaymentProcessorFullDomain, 100, 2);
		$rgeplRequestGatewayEntryPointList->add("https://gw2.".$PaymentProcessorFullDomain, -1, 2);
		$rgeplRequestGatewayEntryPointList->add("https://gw3.".$PaymentProcessorFullDomain, 300, 2);
		
		return $rgeplRequestGatewayEntryPointList;
	}
	
	// }}}
	
	
	// {{{ _parseCardSaveTransactionResponse()

	/**
	 * Parses a response object into name/value pairs.
	 *
	 * @access  private
	 * @param   GatewayOutput             $gateway_output   The response gateway object returned by
	 *                                                      CardSave.
	 * @param   TransactionOutputMessage  $transaction_output_message   The response transaction
	 *                                                                  message returned by CardSave.
	 * @return  array          The parsed key/value pairs from CardSave.
	 */
	function _parseCardSaveTransactionResponse($gateway_output, $transaction_output_message)
	{
		$parsed_response = array();
		
		$parsed_response['status_code'] = $gateway_output->getStatusCode();
		
		$parsed_response['message'] = $gateway_output->getMessage();
		
		// Get status of attempt to authorise the transaction
		if ($gateway_output->getAuthorisationAttempted() != null &&
				$gateway_output->getAuthorisationAttempted()->getHasValue() &&
				$gateway_output->getAuthorisationAttempted()->getValue()) {
			$parsed_response['authorisation_attempted'] = true;
		} else {
			$parsed_response['authorisation_attempted'] = false;
		}
		
		if ($parsed_response['status_code'] == 3) {
			// Store the 3D-Secure transaction information
			$parsed_response['pareq'] = $transaction_output_message->getThreeDSecureOutputData()->getPaREQ();
			$parsed_response['acsurl'] = $transaction_output_message->getThreeDSecureOutputData()->getACSURL();
			$parsed_response['crossreference'] = $transaction_output_message->getCrossReference();
		} else if ($parsed_response['status_code'] == 0) {
			// Store the information about the completed transaction
			$parsed_response['auth_code'] = $transaction_output_message->getAuthCode();
			$parsed_response['cross_reference'] = $transaction_output_message->getCrossReference();
			
			// Store amount received
			if ($transaction_output_message->getAmountReceived() != null &&
					$transaction_output_message->getAmountReceived()->getHasValue()) {
				$parsed_response['amount_received'] = $transaction_output_message->getAmountReceived()->getValue();
	    	}
			
			// Store result of AVS checks
	    	if ($transaction_output_message->getAddressNumericCheckResult() != null &&
					$transaction_output_message->getAddressNumericCheckResult()->getHasValue()) {
	    		$parsed_response['avs_address_numeric_check'] = $transaction_output_message->getAddressNumericCheckResult()->getValue();
	    	} else {
				$parsed_response['avs_address_numeric_check'] = 'Not Checked';
			}
			if ($transaction_output_message->getPostCodeCheckResult() != null &&
					$transaction_output_message->getPostCodeCheckResult()->getHasValue()) {
	    		$parsed_response['avs_postcode_check'] = $transaction_output_message->getPostCodeCheckResult()->getValue();
	    	} else {
				$parsed_response['avs_postcode_check'] = 'Not Checked';
			}
			
	    	// Store result of CV2 check
	    	if ($transaction_output_message->getCV2CheckResult() != null &&
					$transaction_output_message->getCV2CheckResult()->getHasValue()) {
	    		$parsed_response['cv2_check'] = $transaction_output_message->getCV2CheckResult()->getValue();
	    	} else {
				$parsed_response['cv2_check'] = 'Not Checked';
			}
			
	    	// Store result of 3D-Secure authentication/check
	    	if ($transaction_output_message->getThreeDSecureAuthenticationCheckResult() != null &&
					$transaction_output_message->getThreeDSecureAuthenticationCheckResult()->getHasValue()) {
	    		$parsed_response['3d_secure_check'] = $transaction_output_message->getThreeDSecureAuthenticationCheckResult()->getValue();
	    	} else {
				$parsed_response['3d_secure_check'] = 'Not Checked';
			}
			
			// Store information about the gateway entry point(s) used
			$parsed_response['gateway_entry_points'] = '';
			if ($transaction_output_message->getGatewayEntryPoints()) {
				for ($i = 0; $i < $transaction_output_message->getGatewayEntryPoints()->getCount(); $i++) {
					$parsed_response['gateway_entry_points'] .= $transaction_output_message->getGatewayEntryPoints()->getAt($i)->getEntryPointURL() ."\n";
				}
				$parsed_response['gateway_entry_points'] = substr($parsed_response['gateway_entry_points'], 0, strlen($parsed_response['gateway_entry_points']) - 1);
			}
		}
		
		// Check if any information was returned about the card being used
		if (!is_null($transaction_output_message) &&
				$transaction_output_message->getCardTypeData() != null) {
			if ($transaction_output_message->getCardTypeData()->getCardType())
			{
				$parsed_response['card_type'] = $transaction_output_message->getCardTypeData()->getCardType();
			}
			if ($transaction_output_message->getCardTypeData()->getIssuer())
			{
				$parsed_response['card_issuer'] = $transaction_output_message->getCardTypeData()->getIssuer();
			}
		}
		
		return $parsed_response;
	}
	
	// }}}
	
	
	// {{{ _parseCardSaveCardTypeResponse()

	/**
	 * Parses a response object into name/value pairs.
	 *
	 * @access  private
	 * @param   GatewayOutput             $gateway_output   The response gateway object returned by
	 *                                                      CardSave.
	 * @param   TransactionOutputMessage  $transaction_output_message   The response transaction
	 *                                                                  message returned by CardSave.
	 * @return  array          The parsed key/value pairs from CardSave.
	 */
	function _parseCardSaveCardTypeResponse($gateway_output, $transaction_output_message)
	{
		$parsed_response = array();
		
		$parsed_response['status_code'] = $gateway_output->getStatusCode();
		
		$parsed_response['message'] = $gateway_output->getMessage();
		
		// Check if any information was returned about the card being used
		if (!is_null($transaction_output_message) &&
				$transaction_output_message->getCardTypeData() != null) {
			if ($transaction_output_message->getCardTypeData()->getCardType())
			{
				$parsed_response['card_type'] = $transaction_output_message->getCardTypeData()->getCardType();
			}
			if ($transaction_output_message->getCardTypeData()->getIssuer())
			{
				$parsed_response['card_issuer'] = $transaction_output_message->getCardTypeData()->getIssuer();
			}
		}
		
		return $parsed_response;
	}
	
	// }}}
	
	
	// {{{ _identifyErrorMessage()
	
	/**
	 * Identifies the error message to be displayed according to the Status or StatusDetail
	 * information returned by CardSave.
	 *
	 * @access  protected
	 * @param   integer         $status_code      The status code returned by CardSave.
	 * @param   string          $message          The message returned by CardSave.
	 * @param   GatewayOutput   $gateway_output   The response gateway object returned by CardSave.
	 * @param   TransactionOutputMessage  $transaction_output_message   The response transaction
	 *                                                                  message returned by CardSave.
	 * @return  string  The error message for the transaction.
	 */
	function _identifyErrorMessage($status_code, $message, $gateway_output, $transaction_output_message)
	{
		$error_message = null;
		
		if ($status_code == 20) {
			// Duplicate transaction
			if ($gateway_output->getPreviousTransactionResult()->getStatusCode()->getValue() == 0) {
				$error_message = MODULE_PAYMENT_CARDSAVE_TEXT_PREVIOUS_TRANSACTION_PASSED;
			} else {
				$error_message = MODULE_PAYMENT_CARDSAVE_TEXT_PREVIOUS_TRANSACTION_FAILED;
			}
			$error_message = sprintf($error_message, $gateway_output->getPreviousTransactionResult()->getMessage(), $gateway_output->getMessage());
		} else if ($status_code == 5) {
			// Transaction declined
			
			if (!is_null($transaction_output_message)) {
				// Check if the AVS checks failed
				if ($transaction_output_message->getAddressNumericCheckResult() != null &&
						$transaction_output_message->getAddressNumericCheckResult()->getHasValue()) {
					$avs_address_numeric_check = $transaction_output_message->getAddressNumericCheckResult()->getValue();
				} else {
					$avs_address_numeric_check = 'Not Checked';
				}
				if ($transaction_output_message->getPostCodeCheckResult() != null &&
						$transaction_output_message->getPostCodeCheckResult()->getHasValue()) {
					$avs_postcode_check = $transaction_output_message->getPostCodeCheckResult()->getValue();
				} else {
					$avs_postcode_check = 'Not Checked';
				}
				if ($avs_address_numeric_check == 'FAILED' && $avs_postcode_check == 'FAILED') {
					$error_message = MODULE_PAYMENT_CARDSAVE_TEXT_ADDRESS_ERROR;
				} else if ($avs_address_numeric_check == 'FAILED' && $avs_postcode_check == 'PASSED') {
					$error_message = MODULE_PAYMENT_CARDSAVE_TEXT_ADDRESS_NUMERIC_ERROR;
				} else if ($avs_address_numeric_check == 'PASSED' && $avs_postcode_check == 'FAILED') {
					$error_message = MODULE_PAYMENT_CARDSAVE_TEXT_POSTCODE_ERROR;
				}
				
				// Check if the CV2 check failed
				if ($transaction_output_message->getCV2CheckResult() != null &&
						$transaction_output_message->getCV2CheckResult()->getHasValue()) {
					$cv2_check = $transaction_output_message->getCV2CheckResult()->getValue();
				} else {
					$cv2_check = 'Not Checked';
				}
				if ($cv2_check == 'FAILED') {
					$error_message = MODULE_PAYMENT_CARDSAVE_TEXT_CV2_NUMBER_ERROR;
				}
				
				// Check if the 3D-Secure check failed
				if ($transaction_output_message->getThreeDSecureAuthenticationCheckResult() != null &&
						$transaction_output_message->getThreeDSecureAuthenticationCheckResult()->getHasValue()) {
					$threed_secure_check = $transaction_output_message->getThreeDSecureAuthenticationCheckResult()->getValue();
				} else {
					$threed_secure_check = 'Not Checked';
				}
				if ($threed_secure_check == 'FAILED') {
					$error_message = sprintf(MODULE_PAYMENT_CARDSAVE_TEXT_3D_SECURE_DECLINED_MESSAGE, $message);
				}
			}
		} else if ($status_code == 30) {
			// Error occurred
			if ($gateway_output->getErrorMessages()->getCount() > 0)
			{
				for ($i = 0; $i < $gateway_output->getErrorMessages()->getCount(); $i++)
				{
					$error_message = $error_message . "<br />\n" . $gateway_output->getErrorMessages()->getAt($i);
				}
			}
		}
		
		if (is_null($error_message)) {
			// Couldn't identify specific error message
			$error_message = sprintf(MODULE_PAYMENT_CARDSAVE_TEXT_DECLINED_MESSAGE, $message);
		}
		
		return $error_message;
	}
	
	// }}}
	
	
	// {{{ after_order_create()

	/**
	 * Saves the information returned by CardSave for this transaction
	 *
	 * @access  public
	 * @param   int     $zf_order_id   The order id associated with this completed transaction.
	 * @return  none
	 */
	function after_order_create($zf_order_id)
	{
		global $db;
		
		// Save response from cardsave in the database
		$cardsave_response_array = array(
			'cross_reference' => $this->_cardsave_return_values['cross_reference'],
			'tep_order_id' => $zf_order_id,
			'auth_code' => $this->_cardsave_return_values['auth_code'],
			'message' => substr($this->_cardsave_return_values['message'], 0, 255),
			'amount_received' => $this->_cardsave_return_values['amount_received'],
			'avs_address_numeric_check' => $this->_cardsave_return_values['avs_address_numeric_check'],
			'avs_postcode_check' => $this->_cardsave_return_values['avs_postcode_check'],
			'cv2_check' => $this->_cardsave_return_values['cv2_check'],
			'threed_secure_check' => $this->_cardsave_return_values['3d_secure_check'],
			'gateway_entry_points' => substr($this->_cardsave_return_values['gateway_entry_points'], 0, 255)
			);
		tep_db_perform(TABLE_CARDSAVE, $cardsave_response_array);
	}
	
	// }}}
	
	
	// {{{ admin_notification()

	/**
	 * Displays the saved CardSave transaction information in the order details screen in the Admin
	 *
	 * @access  public
	 * @param   int     $zf_order_id  The id of the order for which details should be generated
	 * @return  string  A HTML table detailing the transaction information returned by CardSave
	 */
	function admin_notification($zf_order_id)
	{
		global $db;
		
		$sql = "
			SELECT
				*
			FROM
				" . TABLE_CARDSAVE . "
			WHERE
				tep_order_id = '" . $zf_order_id . "'";
				
		//$pd_sql = $db->Execute($sql);
		$my_var_query = tep_db_query($sql);
		$pd_sql = tep_db_fetch_array($my_var_query);
   
		require(DIR_FS_CATALOG. DIR_WS_MODULES . 'payment/cardsave/cardsave_admin_notification.php');
		
		return $output;
	}
	
	// }}}
	
	
	// {{{ _encodeErrorMessage()

	/**
	 * Encodes tags in an error message by using |lt;| instead of < and |gt;| instead of >
	 * 
	 * Rationale: Unfortunately Zen Cart, rather unintuitively, sanitizes all GET variables
	 * regardless of their content and therefore precludes the passing of HTML tags in error
	 * messages.
	 *
	 * @access  protected
	 * @param   string  $message  The message to be encoded.
	 * @return  string            The encoded message.
	 */
	function _encodeErrorMessage($message)
	{
		$message = str_replace('<', '|lt;|', $message);
		$message = str_replace('>', '|gt;|', $message);
		
		return $message;
	}
	
	// }}}
	
	
	// {{{ get_error()

	/**
	 * Gets the current error message from the URL and returns it for addition to the Message Stack.
	 *
	 * @access  public
	 * @param   none
	 * @return  array  The title and message parts of the error message are returned in a hash.
	 */
	function get_error()
	{
		// Translate Coded Version of HTML error message back into HTML
		// Necessary to get round Zen Cart's interference (sanitisation) of GET Variables.
		$error_message = stripslashes(urldecode($_GET['error']));
		$error_message = str_replace('|lt;|', '<', $error_message);
		$error_message = str_replace('|gt;|', '>', $error_message);
		
		$error = array(
			'title' => MODULE_PAYMENT_CARDSAVE_TEXT_ERROR,
			'error' => $error_message
			);
		
		return $error;
	}
	
	// }}}
	
	
	// {{{ _file_exists_in_include_path()

	/**
	* Check if a file exists in the include path
	*
	* @access   protected
	* @version  1.2.1
	* @author   Aidan Lister <aidan@php.net>
	* @link     http://aidanlister.com/repos/v/function.file_exists_incpath.php
	* @param    string     $file       Name of the file to look for
	* @return   mixed      The full path if file exists, false if it does not
	*/
	function _file_exists_in_include_path($file)
	{
		$paths = explode(PATH_SEPARATOR, get_include_path());
		
		foreach ($paths as $path) {
			// Formulate the absolute path
			$fullpath = $path . DIRECTORY_SEPARATOR . $file;
			
			// Check it
			if (file_exists($fullpath)) {
				return $fullpath;
			}
		}
		
		return false;
	}
	
	// }}}


	// {{{ _useStartDate()

	/**
	 * Examines the list of cards accepted and determines whether at least one of them may need a
	 * start date to be supplied for card processing to take place.
	 *
	 * @access  private
	 * @param   none
	 * @return  bool  Whether at least one of the cards accepted may need a start date to be
	 *                supplied for card processing to take place (boolean true for yes!)
	 */
	function _useStartDate()
	{
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_SOLO == 'Yes') {
			return true;
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_MAESTRO == 'Yes') {
			return true;
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_AMERICAN_EXPRESS == 'Yes') {
			return true;
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _useIssueNumber()

	/**
	 * Examines the list of cards accepted and determines whether at least one of them may need an
	 * issue number to be supplied for card processing to take place.
	 *
	 * @access  private
	 * @param   none
	 * @return  bool  Whether at least one of the cards accepted may need an issue number to be
	 *                supplied for card processing to take place (boolean true for yes!)
	 */
	function _useIssueNumber()
	{
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_SOLO == 'Yes') {
			return true;
		}
		if (MODULE_PAYMENT_CARDSAVE_ACCEPT_MAESTRO == 'Yes') {
			return true;
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _getCardTypeNameForCode()

	/**
	 * Returns the Name of the Card Type for the given Card Type Code. If a surcharge or discount
	 * has been defined for the card type which matches the order, details or the surcharge/discount
	 * are appended to the name. This surcharge/discount is then applied by the
	 * ot_payment_surcharges_discounts Order Total module.
	 *
	 * @access  private
	 * @param   string  $card_type_code                The code of the card Type for which the Name 
	 *                                                 should be returned.
	 * @param   boolean $add_surcharge_discount_info   Whether or not to add details about
	 *                                                 surcharge/discount after the name.
	 * @return  string  The Name of the Card Type (inc possibly any surcharge/discount information).
	 */
	function _getCardTypeNameForCode($card_type_code, $add_surcharge_discount_info = true)
	{
		global $order, $currencies;
		
		$card_type_name = '';
		
		switch ($card_type_code) {
			case 'VISA':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_VISA;
				break;
			case 'MASTERCARD':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_MASTERCARD;
				break;
			case 'VISA_DEBIT':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_VISA_DEBIT;
				break;
			case 'SOLO':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_SOLO;
				break;
			case 'MAESTRO':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_MAESTRO;
				break;
			case 'VISA_ELECTRON':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_VISA_ELECTRON;
				break;
			case 'VISA_PURCHASING':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_VISA_PURCHASING;
				break;
			case 'AMERICAN_EXPRESS':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_AMERICAN_EXPRESS;
				break;
			case 'DINERS_CLUB':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_DINERS_CLUB;
				break;
			case 'JCB':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_JCB;
				break;
			case 'LASER':
				$card_type_name = MODULE_PAYMENT_CARDSAVE_TEXT_LASER;
				break;
			default:
				break;
		}
		
		// Check if the Surcharges/Discounts Order Total module is in use and if so, whether any
		// surcharges/discounts have been specified for the specified card type and should be
		// appended to the card type's name
		if (MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS == 'Yes'
				&& isset($GLOBALS['ot_payment_surcharges_discounts'])
				&& $add_surcharge_discount_info) {
			
			// Check if there are any surcharges/discounts defined for the specified card type
			$table_of_rates = $this->_getSurchargeDiscountTableOfRates($card_type_code);
			
			if ($table_of_rates !== false) {
				// Check if any rate applies to the current order
				$surcharge_or_discount = 
					$GLOBALS['ot_payment_surcharges_discounts']->getSurchargeOrDiscount($table_of_rates);
				
				if (!is_numeric($surcharge_or_discount)) {
					// There was a problem determining the rate
					// Alert the user to the error
					$card_type_name .= ' (' . $surcharge_or_discount . ')';
				} else if ($surcharge_or_discount !== false && $surcharge_or_discount != 0) {
					// A surcharge or discount applies to this card type and order value. Check if
					// some informational text has been defined in the language file
					$surcharge_discount_short_text = trim(constant(
						'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_' . $card_type_code . 
						'_SHORT_' . $_SESSION['languages_id']));
					if (strlen($surcharge_discount_short_text) > 0) {
						$surcharge_or_discount_text = $surcharge_discount_short_text;
					} else {
						// Use default text to inform the user about the surcharge/discount
						if ($surcharge_or_discount < 0) {
							$surcharge_or_discount_text = 
								MODULE_PAYMENT_CARDSAVE_TEXT_DISCOUNT_SHORT;
						} else {
							$surcharge_or_discount_text = 
								MODULE_PAYMENT_CARDSAVE_TEXT_SURCHARGE_SHORT;
						}
					}
					
					$surcharge_or_discount_display_value = 
						$currencies->format($surcharge_or_discount, true, $order->info['currency'], 
						$order->info['currency_value']);
					
					// Alert the user to the amount
					$card_type_name .= ' (' . $surcharge_or_discount_text . ': ' . 
						$surcharge_or_discount_display_value . ')';
				}
			}
		}
		
		return $card_type_name;
	}
	
	// }}}
	
	
	// {{{ _getSurchargeDiscountTableOfRates()

	/**
	 * Checks if a surcharge/discount table of rates has been defined for the specified card type.
	 *
	 * @access  private
	 * @param   string  $card_type_code   The code of the card Type for which the surcharge/discount
	 *                                    table of rates should be returned.
	 * @return  string  The surcharge/discount table of rates for the specified card type.
	 */
	function _getSurchargeDiscountTableOfRates($card_type_code)
	{
		$surcharges_discounts = ereg_replace('[[:space:]]+', '', constant(
			'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_' . $card_type_code));
		if (!is_null($surcharges_discounts) && strlen($surcharges_discounts) > 0) {
			return $surcharges_discounts;
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _cleanString()

	/**
	 * Simple function designed to strip non-standard characters from a string, as required by CardSave
	 * for certain values.
	 *
	 * @access  private
	 * @param   string  $text  The text to be "cleaned".
	 * @return  string         The "cleaned" text.
	 */
	function _cleanString($text)
	{
		$pattern = '|[^a-zA-Z0-9\-\._]+|';
		$text = preg_replace($pattern, '', $text); 
		
		return $text;
	}
	
	// }}}
	
	
	//function check()
	//{
	//	global $db;
	//	
	//	if (!isset($this->_check)) {
	//		//$check_query = $db->Execute("select configuration_value from " . TABLE_CONFIGURATION . " where configuration_key = 'MODULE_PAYMENT_CARDSAVE_STATUS'");
  //    $query_string = "select configuration_value from " . TABLE_CONFIGURATION . " where configuration_key = 'MODULE_PAYMENT_CARDSAVE_STATUS'";
	//	  $my_var_query = tep_db_query($query_string);
  //
  //    $this->_check = tep_db_num_rows($my_var_query);
  //   $this->_check = 1;
	//	}
	//	//return $this->_check;
	//	return $this->_check;
  //  //return 1;
	//}
  
	function check() {
      if (!isset($this->_check)) {
        //$check_query = tep_db_query("select configuration_value from " . TABLE_CONFIGURATION . " where configuration_key = 'MODULE_PAYMENT_PAYPAL_STANDARD_STATUS'");
        $check_query = tep_db_query("select configuration_value from " . TABLE_CONFIGURATION . " where configuration_key = 'MODULE_PAYMENT_CARDSAVE_STATUS'");
        $this->_check = tep_db_num_rows($check_query);
        if ($this->_check > 0) {
            $this->_check = 1;}
      }
      return $this->_check;
    }
    
	function install()
	{
		global $db;
		
		$languages = tep_get_languages();
		
		// General configuration values ////////////////////////////////////////////////////////////
		$background_colour = '#d0d0d0';
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><fieldset style=\"background: " . $background_colour . "; margin-bottom: 1.5em;\"><legend style=\"font-size: 1.4em; font-weight: bold\">General Config</legend><b>Enable CardSave Module', 'MODULE_PAYMENT_CARDSAVE_STATUS', 'Yes', 'Do you want to accept CardSave payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Gateway Account Merchant ID', 'MODULE_PAYMENT_CARDSAVE_GATEWAY_MERCHANT_ID', '', 'The Gateway Account Merchant ID (<strong>not</strong> the Merchant Management System Username).', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Gateway Account Password', 'MODULE_PAYMENT_CARDSAVE_GATEWAY_PASSWORD', '', 'The Gateway Account Password (<strong>not</strong> the Merchant Management System Password).', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Gateway Base URL', 'MODULE_PAYMENT_CARDSAVE_GATEWAY_URL', 'cardsaveonlinepayments.com', 'The base URL for the gateway (Only change if you know what you are doing).', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Gateway Port', 'MODULE_PAYMENT_CARDSAVE_GATEWAY_PORT', '4430', 'The port number for the gateway (Only change if you know what you are doing).', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Merchant Account Accepted Currency/Currencies', 'MODULE_PAYMENT_CARDSAVE_ACCEPTED_CURRENCIES', 'GBP', 'Enter a comma-separated list of the currencies the Merchant can accept. E.g. For two accounts, Pounds Sterling/Euros enter: \'GBP,EUR\'. For one currency enter: \'GBP\'.', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Default Merchant Account\'s Currency', 'MODULE_PAYMENT_CARDSAVE_DEFAULT_CURRENCY', 'GBP', 'Enter the Currency Code for the currency accepted by your Default Merchant Account. E.g. \'GBP\'.', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Authorisation Type', 'MODULE_PAYMENT_CARDSAVE_TRANSACTION_TYPE', 'Immediate Charge', 'Do you want submitted card transactions to be immediately charged, pre-authorised or stored for later? (Consult documentation for an explanation).', '6', '0', 'tep_cfg_select_option(array(\'Immediate Charge\', \'Pre-authorised\', \'Stored\'), ', now())");
		//$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Enable Surcharge/Discount Functionality', 'MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS', 'Yes', 'If enabled, this option will allow you to specify a Single Rate or a Table of Rates for any of the enabled card types, to be used in conjunction with the ot_payment_surcharges_discounts Order Total module, to apply either a surcharge or discount for a card type, dependant on the value of the order.<br /><br />The Rates can be either Specific Values (E.g. 2.00 or -3.50) or Percentages (E.g. 4% or -0.5%) or, <strong>for surcharges only</strong>, a Percentage plus a Specific Value (E.g. 3.4%+0.20).<br /><br /><em>For example</em>: A Single Rate which applies to all Order Values could be specified as &ldquo;2.5%&rdquo; or &ldquo;1.50&rdquo; (without the quotes).<br /><br />The Tables of Rates are comma-separated lists of Limits/Rate pairs. Each Limits/Rate pair consists of an Order Value Range and a Rate, separated by a colon. <br /><br /><em>For example</em>: 1000:2.00,3000:1.50,*:0 <br /><br />In the above example, orders with a Total Value less than 1000 would have a surcharge of 2.00, those from 1000 up to 3000 would have a surcharge of 1.50 and orders of 3000 and above would have no surcharge applied).<br /><br />Notes: An asterix (*) is a wildcard which matches any value, Lower Limits for ranges can be specified by preceding the Upper Limit with a dash (E.g. 300-500).<br /><br />Do you want to enable Surcharges/Discounts?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Enable Surcharge/Discount Functionality', 'MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS', 'Yes', 'If enabled, specifying a Single Rate/Table of Rates for enabled card types, to be used in conjunction with ot_payment_surcharges_discounts Order Total Module, to apply either a surcharge/discount dependant on the order value. Enable Surcharges/Discounts?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, use_function, set_function, date_added) values ('Payment Zone', 'MODULE_PAYMENT_CARDSAVE_ZONE', '0', 'If a zone is selected, this module will only be enabled for the selected zone.<br /><br />Leave set to \"--none--\" if you want to be able to use CardSave for all your customers, regardless of what zone their billing address is in.', '6', '0', 'tep_get_zone_class_title', 'tep_cfg_pull_down_zone_classes(', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, use_function, date_added) values ('Set Order Status', 'MODULE_PAYMENT_CARDSAVE_ORDER_STATUS_ID', '0', 'Set the status of orders made with this payment module to this value.', '6', '0', 'tep_cfg_pull_down_order_statuses(', 'tep_get_order_status_name', now())");
		
		
		// Security configuration values ///////////////////////////////////////////////////////////
		$background_colour = '#eee';
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b></fieldset><fieldset style=\"background: " . $background_colour . "; margin-bottom: 1.5em;\"><legend style=\"font-size: 1.4em; font-weight: bold\">Security Options</legend><b>Store entered details temporarily in session?', 'MODULE_PAYMENT_CARDSAVE_STORE_DETAILS_IN_SESSION', 'Yes', 'Do you want to store the customers card details in the session? If enabled the module will restore the details. Use encrypt for security. If disabled and the customer makes an error they will have to retype the details.', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
    $my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Encryption Keyphrase', 'MODULE_PAYMENT_CARDSAVE_ENCRYPTION_KEYPHRASE', 'Enter your encryption keyphrase here!', 'The keyphrase to be used to encrypt the Card details if they are to be (temporarily) stored in the session.<br /><br />This keyphrase can be <strong>any</strong> random text string you like, just make one up.', '6', '0', now())");
		//$db->Execute("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('AVS (Address Verification) Policy', 'MODULE_PAYMENT_CARDSAVE_AVS_POLICY', 'Only the address numeric check can fail, the postcode check must not fail. (Default)', 'What is the AVS Policy for this store - are any of the AVS checks allowed to fail? (Consult documentation for an explanation).', '6', '0', 'zen_cfg_select_option(array(\'Neither the address numeric check nor the postcode check can fail.\', \'Either the address numeric check can fail or the postcode check can fail but both cannot fail.\', \'Only the address numeric check can fail, the postcode check must not fail. (Default)\', \'Only the postcode check can fail, the address numeric check must not fail.\', \'Both the address numeric and postcode checks can fail.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('AVS (Address Verification) Policy', 'MODULE_PAYMENT_CARDSAVE_AVS_POLICY', 'Only address numeric check CAN fail(Default)', 'What is the AVS Policy for this store - are any of the AVS checks allowed to fail? (Consult documentation for an explanation).', '6', '0', 'tep_cfg_select_option(array(\'Both checks CANNOT fail.\', \'Address numeric/postcode check CAN fail but NOT BOTH.\', \'Only address numeric check CAN fail(Default)\', \'Only postcode check CAN fail\', \'Address numeric/postcode check CAN fail.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('AVS Address Numeric Partial Matches', 'MODULE_PAYMENT_CARDSAVE_AVS_POLICY_ADDRESS_NUMERIC_PARTIAL_MATCH', 'Partial matches should be treated as a fail. (Default)', 'What should happen when an address numeric returns only a partial match?', '6', '0', 'tep_cfg_select_option(array(\'Partial matches should be treated as a fail. (Default)\', \'Partial matches should be treated as a pass.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('AVS Postcode Partial Matches', 'MODULE_PAYMENT_CARDSAVE_AVS_POLICY_POSTCODE_PARTIAL_MATCH', 'Partial matches should be treated as a fail. (Default)', 'What should happen when a postcode returns only a partial match?', '6', '0', 'tep_cfg_select_option(array(\'Partial matches should be treated as a fail. (Default)\', \'Partial matches should be treated as a pass.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('AVS Results Unknown', 'MODULE_PAYMENT_CARDSAVE_AVS_POLICY_RESULTS_UNKNOWN', 'Both checks having unknown results should be treated as a fail. (Default)', 'What should happen if both the results of the address numeric and postcode checks are unknown?', '6', '0', 'tep_cfg_select_option(array(\'Both checks having unknown results should be treated as a fail. (Default)\', \'Both checks have unknown results should be treated as a pass.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Request CV2 Number', 'MODULE_PAYMENT_CARDSAVE_USE_CV2', 'Yes', 'Do you want to ask the customer for the card\'s CV2 number?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('CV2 (Card Verification Number) Policy', 'MODULE_PAYMENT_CARDSAVE_CV2_POLICY', 'The CV2 check must not fail. (Default)', 'What is the CV2 Policy for this store - is the CV2 check allowed to fail? (Consult documentation for an explanation).', '6', '0', 'tep_cfg_select_option(array(\'The CV2 check must not fail. (Default)\', \'The CV2 check can fail.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('CV2 Results Unknown', 'MODULE_PAYMENT_CARDSAVE_CV2_POLICY_RESULTS_UNKNOWN', 'The CV2 check having unknown results should be treated as a fail. (Default)', 'What should happen if the result of the CV2 check is unknown?', '6', '0', 'tep_cfg_select_option(array(\'The CV2 check having unknown results should be treated as a fail. (Default)\', \'The CV2 check having unknown results should be treated as a pass.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('3D-Secure Policy', 'MODULE_PAYMENT_CARDSAVE_3D_SECURE_POLICY', 'Use 3D-Secure whenever possible. (Default)', 'What is the 3D-Secure Policy for this store - should 3D-Secure checks be skipped if possible? (Certain card types will always cause 3D-Secure to be used).', '6', '0', 'tep_cfg_select_option(array(\'Use 3D-Secure whenever possible. (Default)\', \'Only use 3D-Secure for card types that absolutely require it.\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Disable Autocomplete for Card Number field', 'MODULE_PAYMENT_CARDSAVE_DISABLE_CARD_NUMBER_AUTOCOMPLETE', 'Yes', 'Do you want to disable the autocomplete functionality of certain browsers for the Card Number field? (This prevents the browser from automatically entering the user\'s Card Number).', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Disable Autocomplete for CV2 field (if CV2 enabled!)', 'MODULE_PAYMENT_CARDSAVE_DISABLE_CV2_AUTOCOMPLETE', 'Yes', 'Do you want to disable the autocomplete functionality of certain browsers for the CV2 field? (This prevents the browser from automatically entering the user\'s CV2 Number).', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		
		
		// Card configuration values ///////////////////////////////////////////////////////////////
		$background_colour = '#d0d0d0';
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b></fieldset><fieldset style=\"background: " . $background_colour . "; margin-bottom: 1.5em;\"><legend style=\"font-size: 1.4em; font-weight: bold\">Card Type Configs</legend><b>Visa Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA', 'Yes', 'Do you want to Accept Visa Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Visa Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA', '', 'If there are surcharge(s) or discount(s) for Visa Card payments, enter a Rate or a Table of Rates here.', '6', '0', now())");
		
		// Language text for Visa Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;2% Surcharge&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Visa Card Surcharge @ 2%&rdquo;)', '6', '0', now())");
		}
		
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>MasterCard Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_MASTERCARD', 'Yes', 'Do you want to Accept MasterCard Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('MasterCard Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MASTERCARD', '', 'If there are surcharge(s) or discount(s) for MasterCard Card payments, enter a Rate or a Table of Rates here.', '6', '0', now())");
		
		// Language text for MasterCard Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MASTERCARD_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;2% Surcharge&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MASTERCARD_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;MasterCard Card Surcharge @ 2%&rdquo;)', '6', '0', now())");
		}
		
		// Code for "Visa Debit" is "VISA_DEBIT"
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>Visa Debit Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_DEBIT', 'Yes', 'Do you want to Accept Visa Debit Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Visa Debit Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_DEBIT', '', 'If there are surcharge(s) or discount(s) for Visa Debit Card payments, enter a Rate or a Table of Rates here. <br /><br />Please Note: Most policies forbid surcharges for debit cards!', '6', '0', now())");
		
		// Language text for Visa Debit Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_DEBIT_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;&pound;0.50 Discount&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_DEBIT_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Visa Debit Card Discount&rdquo;)', '6', '0', now())");
		}
		
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>Solo Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_SOLO', 'Yes', 'Do you want to Accept Solo Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Solo Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_SOLO', '', 'If there are surcharge(s) or discount(s) for Solo Card payments, enter a Rate or a Table of Rates here. <br /><br />Please Note: Most policies forbid surcharges for debit cards!', '6', '0', now())");
		
		// Language text for Solo Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_SOLO_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;&pound;0.50 Discount&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_SOLO_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Solo Card Discount&rdquo;)', '6', '0', now())");
		}
		
		// Code for "Maestro" is "MAESTRO"
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>Maestro Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_MAESTRO', 'Yes', 'Do you want to Accept Maestro Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Maestro Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MAESTRO', '', 'If there are surcharge(s) or discount(s) for Maestro Card payments, enter a Rate or a Table of Rates here. <br /><br />Please Note: Most policies forbid surcharges for debit cards!', '6', '0', now())");
		
		// Language text for Maestro Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MAESTRO_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;&pound;0.50 Discount&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MAESTRO_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Maestro Card Discount&rdquo;)', '6', '0', now())");
		}
		
		// Code for "Visa Electron" is "VISA_ELECTRON"
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>Visa Electron (UK Electron) Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_ELECTRON', 'Yes', 'Do you want to Accept Visa Electron Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Visa Electron Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_ELECTRON', '', 'If there are surcharge(s) or discount(s) for Visa Electron Card payments, enter a Rate or a Table of Rates here. <br /><br />Please Note: Most policies forbid surcharges for debit cards!', '6', '0', now())");
		
		// Language text for Visa Electron Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_ELECTRON_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;&pound;0.50 Discount&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_ELECTRON_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Visa Electron Card Discount&rdquo;)', '6', '0', now())");
		}
		
		// Code for "Visa Purchasing" is "VISA_PURCHASING"
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>Visa Purchasing Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_PURCHASING', 'Yes', 'Do you want to Accept Visa Purchasing Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Visa Debit Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_PURCHASING', '', 'If there are surcharge(s) or discount(s) for Visa Purchasing Card payments, enter a Rate or a Table of Rates here. <br /><br />Please Note: Most policies forbid surcharges for debit cards!', '6', '0', now())");
		
		// Language text for Visa Debit Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_PURCHASING_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;&pound;0.50 Discount&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_PURCHASING_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Visa Purchasing Card Discount&rdquo;)', '6', '0', now())");
		}
		
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>American Express Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_AMERICAN_EXPRESS', 'No', 'Do you want to Accept American Express Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('American Express Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_AMERICAN_EXPRESS', '', 'If there are surcharge(s) or discount(s) for American Express Card payments, enter a Rate or a Table of Rates here.', '6', '0', now())");
		
		// Language text for American Express Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_AMERICAN_EXPRESS_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;4% Surcharge&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_AMERICAN_EXPRESS_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;American Express Card Surcharge @ 4%&rdquo;)', '6', '0', now())");
		}
		
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>Diners Club Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_DINERS_CLUB', 'No', 'Do you want to Accept Diners Club Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Diners Club Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_DINERS_CLUB', '', 'If there are surcharge(s) or discount(s) for Diners Club Card payments, enter a Rate or a Table of Rates here.', '6', '0', now())");
		
		// Language text for Diners Club Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_DINERS_CLUB_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;2% Surcharge&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_DINERS_CLUB_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Diners Club Card Surcharge @ 2%&rdquo;)', '6', '0', now())");
		}
		
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>JCB Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_JCB', 'No', 'Do you want to Accept JCB Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('JCB Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_JCB', '', 'If there are surcharge(s) or discount(s) for JCB Club Card payments, enter a Rate or a Table of Rates here.', '6', '0', now())");
		
		// Language text for JCB Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_JCB_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;2% Surcharge&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_JCB_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;JCB Card Surcharge @ 2%&rdquo;)', '6', '0', now())");
		}
		
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b><hr /><b>Laser Card Payments', 'MODULE_PAYMENT_CARDSAVE_ACCEPT_LASER', 'No', 'Do you want to Accept Laser Card Payments?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Laser Card Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_LASER', '', 'If there are surcharge(s) or discount(s) for Laser Club Card payments, enter a Rate or a Table of Rates here.', '6', '0', now())");
		
		// Language text for Laser Surcharges/Discounts
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Short Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_LASER_SHORT_" . $languages[$i]['id'] . "', '', 'Short Descriptive Text to be added after card\'s title in the Card Type selection gadget (E.g. &ldquo;2% Surcharge&rdquo;)', '6', '0', now())");
			$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('" . tep_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) . "&nbsp;Long Text', 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_LASER_LONG_" . $languages[$i]['id'] . "', '', 'Longer Descriptive Text for Order Total Summary Line (E.g. &ldquo;Laser Card Surcharge @ 2%&rdquo;)', '6', '0', now())");
		}
		
		
		// Display configuration values ////////////////////////////////////////////////////////////
		$background_colour = '#eee';
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b></fieldset><fieldset style=\"background: " . $background_colour . "; margin-bottom: 1.5em;\"><legend style=\"font-size: 1.4em; font-weight: bold\">Display Options</legend><b>Show icons of Cards Accepted', 'MODULE_PAYMENT_CARDSAVE_SHOW_CARDS_ACCEPTED', 'Yes', 'Do you want to show icons for each Credit/Debit Card accepted?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Start/Expiry Month Format', 'MODULE_PAYMENT_CARDSAVE_SELECT_MONTH_FORMAT', '%m - %B', 'Enter a valid strftime format code here to be used within the Start and Expiry Date Month Selection gadgets.', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Start/Expiry Year Format', 'MODULE_PAYMENT_CARDSAVE_SELECT_YEAR_FORMAT', '%Y', 'Enter a valid strftime format code here to be used within the Start and Expiry Date Year Selection gadgets.', '6', '0', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Show Message about Surcharges/Discounts', 'MODULE_PAYMENT_CARDSAVE_ENABLE_CUSTOM_SURCHARGES_DISCOUNTS_MESSAGE', 'Yes', 'If using the Surcharges/Discounts functionality, it is beneficial to give the customer information about your policy. If enabled the message defined in the Languages Definition file will be displayed above the Card Type gadget. Enable?', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, date_added) values ('Sort Order of Display.', 'MODULE_PAYMENT_CARDSAVE_SORT_ORDER', '0', 'The Sort Order of Display determines what order the installed payment modules are displayed in. The module with the lowest Sort Order is displayed first (towards the top).', '6', '0', now())");
		
		
		// Miscellaneous options ///////////////////////////////////////////////////////////////////
		$background_colour = '#d0d0d0';
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b></fieldset><fieldset style=\"background: " . $background_colour . "; margin-bottom: 1.5em;\"><legend style=\"font-size: 1.4em; font-weight: bold\">Misc. Options</legend><b>Enable Test Mode', 'MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED', 'Yes', 'If test mode is enabled, a list of Test Cases is displayed at checkout which can be used to quickly and easily test out various card scenarios.', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Enable Debugging Output', 'MODULE_PAYMENT_CARDSAVE_DEBUGGING_ENABLED', 'No', 'If enabled, this will cause the Cart to stop after attempting to complete the transaction with CardSave. The data sent and received will be output instead of the Checkout payment page.<br /><br />DON\'T ENABLE UNLESS YOU KNOW WHAT YOU ARE DOING!', '6', '0', 'tep_cfg_select_option(array(\'Yes\', \'No\'), ', now())");
		$my_var_query = tep_db_query("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('</b></fieldset><img src=\"" . DIR_WS_ADMIN . "/images/ceon_button_logo.png\" alt=\"Made by Ceon. &copy; 2008 Ceon\" align=\"right\" style=\"margin: 1em 0.2em;\"/><br />Module &copy; 2008 Ceon<p style=\"display: none\">', 'MODULE_PAYMENT_CARDSAVE_MADE_BY_CEON', '" . $this->version . "', '', '6', '0', 'tep_draw_hidden_field(\'made_by_ceon\' . ', now())");
	}

	function remove()
	{
		global $db;
		$my_var_query = tep_db_query("delete from " . TABLE_CONFIGURATION . " where configuration_key in ('" . implode("', '", $this->keys()) . "')");
	}

	function keys()
	{
		$languages = tep_get_languages();
		
		$keys = array(
			'MODULE_PAYMENT_CARDSAVE_STATUS',
			'MODULE_PAYMENT_CARDSAVE_GATEWAY_MERCHANT_ID',
			'MODULE_PAYMENT_CARDSAVE_GATEWAY_PASSWORD',
			'MODULE_PAYMENT_CARDSAVE_GATEWAY_URL',
			'MODULE_PAYMENT_CARDSAVE_GATEWAY_PORT',
			'MODULE_PAYMENT_CARDSAVE_ACCEPTED_CURRENCIES',
			'MODULE_PAYMENT_CARDSAVE_DEFAULT_CURRENCY',
			'MODULE_PAYMENT_CARDSAVE_TRANSACTION_TYPE',
			'MODULE_PAYMENT_CARDSAVE_ENABLE_SURCHARGES_DISCOUNTS',
			'MODULE_PAYMENT_CARDSAVE_ZONE',
			'MODULE_PAYMENT_CARDSAVE_ORDER_STATUS_ID',
			'MODULE_PAYMENT_CARDSAVE_STORE_DETAILS_IN_SESSION',
			'MODULE_PAYMENT_CARDSAVE_ENCRYPTION_KEYPHRASE',
			'MODULE_PAYMENT_CARDSAVE_AVS_POLICY',
			'MODULE_PAYMENT_CARDSAVE_AVS_POLICY_ADDRESS_NUMERIC_PARTIAL_MATCH',
			'MODULE_PAYMENT_CARDSAVE_AVS_POLICY_POSTCODE_PARTIAL_MATCH',
			'MODULE_PAYMENT_CARDSAVE_AVS_POLICY_RESULTS_UNKNOWN',
			'MODULE_PAYMENT_CARDSAVE_USE_CV2',
			'MODULE_PAYMENT_CARDSAVE_CV2_POLICY',
			'MODULE_PAYMENT_CARDSAVE_CV2_POLICY_RESULTS_UNKNOWN',
			'MODULE_PAYMENT_CARDSAVE_3D_SECURE_POLICY',
			'MODULE_PAYMENT_CARDSAVE_DISABLE_CARD_NUMBER_AUTOCOMPLETE',
			'MODULE_PAYMENT_CARDSAVE_DISABLE_CV2_AUTOCOMPLETE',
			'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA',
			'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA');
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_MASTERCARD';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MASTERCARD';
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MASTERCARD_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MASTERCARD_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_DEBIT';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_DEBIT'; // Code for "Visa Debit" is "VISA_DEBIT"
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_DEBIT_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_DEBIT_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_SOLO';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_SOLO';
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_SOLO_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_SOLO_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_MAESTRO';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MAESTRO';
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MAESTRO_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_MAESTRO_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_ELECTRON';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_ELECTRON'; // Code for "Visa Electron" is "VISA_ELECTRON"
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_ELECTRON_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_ELECTRON_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_VISA_PURCHASING';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_PURCHASING'; // Code for "Visa Purchasing" is "VISA_PURCHASING"
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_PURCHASING_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_VISA_PURCHASING_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_AMERICAN_EXPRESS';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_AMERICAN_EXPRESS';
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_AMERICAN_EXPRESS_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_AMERICAN_EXPRESS_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_DINERS_CLUB';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_DINERS_CLUB';
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_DINERS_CLUB_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_DINERS_CLUB_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_JCB';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_JCB';
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_JCB_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_JCB_LONG_' . $languages[$i]['id'];
		}
		
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_ACCEPT_LASER';
		$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_LASER';
		
		for ($i = 0, $n = sizeof($languages); $i < $n; $i++) {
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_LASER_SHORT_' . $languages[$i]['id'];
			$keys[] = 'MODULE_PAYMENT_CARDSAVE_SURCHARGES_DISCOUNTS_LASER_LONG_' . $languages[$i]['id'];
		}
		
		$remaining_keys = array(
			'MODULE_PAYMENT_CARDSAVE_SHOW_CARDS_ACCEPTED',
			'MODULE_PAYMENT_CARDSAVE_SELECT_MONTH_FORMAT',
			'MODULE_PAYMENT_CARDSAVE_SELECT_YEAR_FORMAT',
			'MODULE_PAYMENT_CARDSAVE_ENABLE_CUSTOM_SURCHARGES_DISCOUNTS_MESSAGE',
			'MODULE_PAYMENT_CARDSAVE_SORT_ORDER',
			'MODULE_PAYMENT_CARDSAVE_TEST_MODE_ENABLED',
			'MODULE_PAYMENT_CARDSAVE_DEBUGGING_ENABLED',
			'MODULE_PAYMENT_CARDSAVE_MADE_BY_CEON'
			);
		
		$keys = array_merge($keys, $remaining_keys);
		
		return $keys;
	}
}

// }}}

?>
